/*browser:true*/
/*global define*/

define([
  "jquery",
  "Magento_Payment/js/view/payment/cc-form",
  "Magento_Checkout/js/model/url-builder",
  "mage/storage",
  "Magento_Checkout/js/model/full-screen-loader",
  "ko",
  "Magento_Checkout/js/model/quote",
  "Magento_Checkout/js/action/redirect-on-success",
  "Magento_CheckoutAgreements/js/model/agreement-validator",
  "Magento_Checkout/js/model/payment/additional-validators",
  "Magento_Checkout/js/action/set-payment-information",
  "mage/url",
], function (
  $,
  Component,
  urlBuilder,
  storage,
  fullScreenLoader,
  ko,
  quote,
  redirectOnSuccessAction,
  agreementValidator,
  additionalValidators,
  setPaymentInformation,
  mageUrl
) {
  "use strict";
  return Component.extend({
    defaults: {
      showPostCode: window.checkoutConfig.payment.revolut_form.showPostCode,
      revolutCard: ko.observable(null),
      isPaymentTotalUpdateAllowed: ko.observable(true),
      updateOrderTotals: ko.observable(Promise.resolve()),
      publicId: ko.observable(null),
      publicKey: ko.observable(null),
      paymentCurrency: ko.observable(null),
      paymentAmount: ko.observable(null),
      errorCreateRevolutOrder: ko.observable(null),
      retryOrderPlaceActionCount: ko.observable(0),
      errorWidgetTarget: "#show-error-card-error",
      orderCanceledMsg:
        "An error occurred while processing your order. Your payment has been canceled, please review order details and try again.",
      retryOrderPlaceActionLimit: 3,
      retryOrderPlaceActionDelay: 2,
    },

    initialize: function () {
      $(`script[src="${window.checkoutConfig.payment.revolut_form.revolutSdk}"]`).attr("id", "revolut-checkout");

      this._super();
      let self = this;

      this.totals.subscribe(function (data) {
        if (!this.publicId() || !this.isPaymentTotalUpdateAllowed()) {
          return;
        }

        fullScreenLoader.startLoader();

        storage
          .post(
            urlBuilder.createUrl("/revolut/update/order", {}),
            JSON.stringify({
              revolut_public_id: this.publicId(),
            }),
            true
          )
          .done(function (response) {
            self.setPaymentDetails(response);
            self.createRevolutWidgetInstance(response);
          });
      }, this);
    },

    createRevolutForm: function (errorMsg = "") {
      let self = this;
      fullScreenLoader.startLoader();
      if (!window.createRevolutForm) {
        window.createRevolutForm = storage.post(urlBuilder.createUrl("/revolut/create/order", {}));
      }

      window.createRevolutForm.done(function (response) {
        self.setPaymentDetails(response);
        self.createRevolutWidgetInstance(response);
        if (errorMsg && typeof errorMsg !== "object") {
          self.handleError({
            message: errorMsg,
          });
        }
        self.isPlaceOrderActionAllowed(true);
      });
    },

    selectRevolutPaymentMethod: function () {
      $(this.errorWidgetTarget).hide();
      this.createRevolutForm();
    },

    setPaymentDetails: function (response) {
      let data = JSON.parse(response);
      if (!data.success) {
        return;
      }

      if (
        data.available_card_brands &&
        Array.isArray(data.available_card_brands) &&
        data.available_card_brands.includes("AMEX")
      ) {
        $(".revolut-amex-logo").show();
      }

      this.paymentAmount(data.amount);
      this.paymentCurrency(data.currency);
      this.publicKey(data.public_key);
      this.publicId(data.public_id);
    },

    initObservable: function () {
      this._super();
      return this;
    },

    totals: quote.getTotals(),

    placeOrder: function (data, event) {
      let self = this;

      if (event) {
        event.preventDefault();
      }

      if (!this.validate() || !this.customValidations()) {
        return false;
      }

      if (this.errorCreateRevolutOrder()) {
        this.messageContainer.addErrorMessage({
          message: $.mage.__(this.errorCreateRevolutOrder()),
        });

        fullScreenLoader.stopLoader();

        return false;
      }

      this.isPlaceOrderActionAllowed(false);
      fullScreenLoader.startLoader();

      $.when(setPaymentInformation(this.messageContainer, this.getData()))
        .done(function () {
          self.updateOrderTotals().then(function () {
            self.revolutCard().submit({
              email: self.getBillingEmail(),
              phone: self.getBillingPhone(),
              name: self.getBillingName(),
              billingAddress: self.getFormattedBillingAddress(),
            });
          });
        })
        .fail(function (response) {
          fullScreenLoader.stopLoader();
        });

      return true;
    },

    getFormattedBillingAddress: function () {
      let address = quote.billingAddress();

      if (!address) return null;

      let streetLine = undefined;
      if (address.street && address.street.length > 0) {
        streetLine = address.street[0];
      }

      return {
        countryCode: address.countryId,
        region: address.regionCode,
        city: address.city,
        streetLine1: streetLine,
        postcode: address.postcode,
      };
    },

    getBillingName: function () {
      let billingAddress = quote.billingAddress();

      if (!billingAddress) return null;

      if (billingAddress.firstname && billingAddress.lastname) {
        return `${billingAddress.firstname} ${billingAddress.lastname}`;
      }

      window.location.reload();

      return undefined;
    },

    getBillingEmail: function () {
      if (quote.guestEmail) {
        return quote.guestEmail;
      } else if (window.checkoutConfig.customerData && window.checkoutConfig.customerData.email) {
        return window.checkoutConfig.customerData.email;
      }

      return undefined;
    },

    getBillingPhone: function () {
      let billingAddress = quote.billingAddress();
      if (!billingAddress) return null;

      if (billingAddress.telephone) return billingAddress.telephone;

      let shippingAddress = quote.shippingAddress();
      if (shippingAddress.telephone) return shippingAddress.telephone;

      return undefined;
    },

    customValidations: function (elm) {
      return agreementValidator.validate() && additionalValidators.validate();
    },

    context: function () {
      return this;
    },

    getCode: function () {
      return "revolut_form";
    },

    isActive: function () {
      return true;
    },

    getData: function () {
      return {
        method: this.getCode(),
        additional_data: {
          publicId: this.publicId(),
        },
      };
    },

    handleError: function (messages) {
      if (
        !messages ||
        (Array.isArray(messages) && !messages.length && this.orderCanceledMsg == $(this.errorWidgetTarget).text())
      ) {
        return;
      }

      this.isPaymentTotalUpdateAllowed(true);
      $(this.errorWidgetTarget).hide();

      let errorMessage = "";

      if (messages && Array.isArray(messages)) {
        let errorMessages = [];
        messages.forEach(function (message) {
          if (message) {
            errorMessages.push(message.message);
          }
        });
        errorMessage = errorMessages.join(", ");
      } else {
        if (messages.message) {
          errorMessage = messages.message;
        }
      }

      if (errorMessage) {
        $(this.errorWidgetTarget).text(errorMessage);
        $(this.errorWidgetTarget).show();
      }

      this.isPlaceOrderActionAllowed(true);
      fullScreenLoader.stopLoader();
    },

    handleSuccess: function () {
      let self = this;
      this.isPaymentTotalUpdateAllowed(false);
      self.updateOrderTotals().then(function () {
        self
          .getPlaceOrderDeferredObject()
          .fail(function (response) {
            self.logError(response);
            self.retryingOrderPlace();

            if (self.retryOrderPlaceActionCount() < self.retryOrderPlaceActionLimit) {
              let retryDelay = self.retryOrderPlaceActionCount() * self.retryOrderPlaceActionDelay * 1000;

              fullScreenLoader.startLoader();
              self.isPlaceOrderActionAllowed(false);
              setTimeout(function () {
                fullScreenLoader.stopLoader();
                self.handleSuccess();
              }, retryDelay);

              return;
            }

            self.cancelPayment();
            self.isPlaceOrderActionAllowed(false);
            self.retryOrderPlaceActionCount(0);
          })
          .done(function () {
            self.afterPlaceOrder();
            redirectOnSuccessAction.execute();
          });
      });
    },

    retryingOrderPlace: function () {
      let retryOrderPlaceAction = this.retryOrderPlaceActionCount() + 1;
      this.retryOrderPlaceActionCount(retryOrderPlaceAction);
    },

    cancelPayment: function () {
      const self = this;
      fullScreenLoader.startLoader();

      fetch(mageUrl.build(urlBuilder.createUrl("/revolut/cancel/order", {})), {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          revolut_public_id: this.publicId(),
        }),
      }).then((response) => {
        window.createRevolutForm = null;
        self.createRevolutForm(self.orderCanceledMsg);
        fullScreenLoader.stopLoader();
      });
    },

    logError: function (errorMsg) {
      fetch(mageUrl.build(urlBuilder.createUrl("/log/error", {})), {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(errorMsg),
      });
    },

    handleValidate: function () {
      let self = this;
      this.isPaymentTotalUpdateAllowed(true);
      return new Promise(function (resolve, reject) {
        if (!self.validate() || !self.customValidations()) {
          return reject(self.orderCanceledMsg);
        }

        if (self.isPlaceOrderActionAllowed()) {
          $.when(setPaymentInformation(self.messageContainer, self.getData()))
            .done(function () {
              return resolve(true);
            })
            .fail(function (response) {
              return reject(self.orderCanceledMsg);
            });

          return;
        }

        fullScreenLoader.stopLoader();
        reject("Place Order action is not allowed.");
      });
    },

    handleCancel: function () {
      this.isPaymentTotalUpdateAllowed(true);
      fullScreenLoader.stopLoader();
    },
  });
});
