<?php

namespace Revolut\Payment\Observer;

use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\View\LayoutInterface;
use Magento\Framework\Event\Observer;

use Revolut\Payment\Model\RevolutOrderFactory;
use Revolut\Payment\Model\ResourceModel\RevolutOrder;
use Revolut\Payment\Helper\DataHelper;
use Revolut\Payment\Helper\Logger;
use Revolut\Payment\Helper\ConfigHelper;

class OnepageControllerSuccessAction implements ObserverInterface
{
    /**
     * @var LayoutInterface
     */
    private $layout;

    /**
     * @var DataHelper
     */
    protected $dataHelper;

    /**
     * @var Logger
     */
    protected $loggerRevolut;

    /**
     * @var RevolutOrderFactory
     */
    protected $revolutOrderFactory;

    /**
     * @var RevolutOrder
     */
    protected $revolutOrderResourceModel;


    /**
     * @var ConfigHelper
     */
    protected $configHelper;


    public function __construct(
        LayoutInterface $layout,
        Logger $logger,
        RevolutOrderFactory $revolutOrderFactory,
        RevolutOrder $revolutOrderResourceModel,
        DataHelper $dataHelper,
        ConfigHelper $configHelper
    ) {
        $this->layout = $layout;
        $this->loggerRevolut = $logger;
        $this->dataHelper = $dataHelper;
        $this->revolutOrderFactory = $revolutOrderFactory;
        $this->configHelper = $configHelper;
        $this->revolutOrderResourceModel = $revolutOrderResourceModel;
    }

    /**
     * Execute observer
     *
     * @param Observer $observer
     * @return void|null
     */
    public function execute(
        Observer $observer
    ) {
        try {
            $this->loggerRevolut->debug("banner visibility: " . (bool)$this->configHelper->hideCashbackBanner());
            $block = $this->layout->getBlock('revolut.reward.banner.block');

            if (!$block || !($block instanceof \Magento\Framework\View\Element\BlockInterface)) {
                return;
            }

            if ((bool)$this->configHelper->hideCashbackBanner()) {
                $block->setShowBanner(false);
                return;
            }

            $orderIds = $observer->getEvent()->getOrderIds();
        
            if (empty($orderIds) || !is_array($orderIds)) {
                return;
            }
    
            $orderId = reset($orderIds);
    
            if (empty($orderId)) {
                return;
            }
    
            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
            // @phpstan-ignore-next-line
            $order = $objectManager->create('\Magento\Sales\Model\Order')->load($orderId);
    
            $billingAddress = $order->getBillingAddress();
            $phone = false;

            if ($billingAddress) {
                $phone = $billingAddress->getTelephone();
            }
            
            $email = $order->getCustomerEmail();
            $incrementOrderId = $order->getIncrementId();
    
    
            $revolutOrderModel = $this->revolutOrderFactory->create();
            $this->revolutOrderResourceModel->load($revolutOrderModel, $incrementOrderId, 'increment_order_id');
            $revolutOrderId = $revolutOrderModel->getData('revolut_order_id');
            $this->loggerRevolut->debug("revolutOrderId: " . $revolutOrderId);
    
            if (empty($revolutOrderId) || empty($phone) || empty($email)) {
                $block->setShowBanner(false);
                return;
            }
    
            $paymentToken = (string) $this->dataHelper->getPaymentToken($revolutOrderId);
            $this->loggerRevolut->debug("paymentToken: " . $paymentToken);
    
            if (empty($paymentToken)) {
                $block->setShowBanner(false);
                return;
            }
    
            $cashbackOffer = $this->dataHelper->getCashbackOffer($paymentToken);
            $this->loggerRevolut->debug("cashbackOffer: " . \json_encode($cashbackOffer));
    
            if (empty($cashbackOffer)) {
                $block->setShowBanner(false);
                return;
            }
    
            $cashbackCurrency = $cashbackOffer['cashbackCurrency'];
            $cashbackAmount = $cashbackOffer['cashbackAmount'];
    
            $storeManager = \Magento\Framework\App\ObjectManager::getInstance()
            ->get(\Magento\Store\Model\StoreManagerInterface::class);

            if (! $storeManager) {
                $block->setShowBanner(false);
                return;
            }

            $storeCode = $storeManager->getStore()->getCode();
    
            $currencyModel = $objectManager->create('Magento\Directory\Model\Currency');
            
            if (! $currencyModel) {
                $block->setShowBanner(false);
                return;
            }

            $cashbackCurrencySymbol = $currencyModel->load($cashbackCurrency)->getCurrencySymbol();
            
            if (empty($cashbackCurrencySymbol)) {
                $cashbackCurrencySymbol = $cashbackCurrency;
            }

            $formattedPrice = $currencyModel->format($cashbackAmount, ['symbol' => $cashbackCurrencySymbol, 'precision'=> 2], false, false);
    
            $block->setShowBanner(true);
            $block->setCashbackFormattedPrice($formattedPrice);
            $block->setStoreCode($storeCode);
            $block->setOrderId($orderId);
            $block->setPhone($phone);
            $block->setEmail($email);
        } catch (\Exception $e) {
            if ($block instanceof \Magento\Framework\View\Element\BlockInterface) {
                $block->setShowBanner(false);
            }
        
            $this->loggerRevolut->debug("An error occured while checking cashback offer: " . $e->getMessage());
        }
    }
}
