<?php

namespace Revolut\Payment\Model;

use Revolut\Payment\Helper\ConfigHelper;
use Revolut\Payment\Helper\DataHelper;
use Revolut\Payment\Helper\Logger;

abstract class RevolutAbstractPaymentMethod extends \Magento\Payment\Model\Method\AbstractMethod
{
    const CODE = 'revolut';
    protected $_code = self::CODE;
    protected $_canAuthorize = true;
    protected $_canCapture = true;
    protected $_isGateway = true;
    protected $_canRefund = true;
    protected $_canRefundInvoicePartial = true;
    protected $_canVoid = true;
    protected $_canUseInternal = false;
    /**
     * @var \Magento\Framework\Module\Manager
     */
    protected $moduleManager;

    /**
     * @var ConfigHelper
     */
    protected $configHelper;

    /**
     * @var RevolutOrderFactory
     */
    protected $revolutOrderFactory;

    /**
     * @var DataHelper
     */
    protected $dataHelper;

    /**
     * @var \Magento\Framework\Json\Helper\Data
     */
    protected $jsonHelper;

    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    protected $managerInterface;

    /**
     * @var Logger
     */
    protected $loggerRevolut;

    /**
     * RevolutPayForm constructor.
     *
     * @param \Magento\Framework\Model\Context                             $context
     * @param \Magento\Framework\Registry                                  $registry
     * @param \Magento\Framework\Api\ExtensionAttributesFactory            $extensionFactory
     * @param \Magento\Framework\Api\AttributeValueFactory                 $customAttributeFactory
     * @param \Magento\Payment\Helper\Data                                 $paymentData
     * @param \Magento\Framework\App\Config\ScopeConfigInterface           $scopeConfig
     * @param \Magento\Payment\Model\Method\Logger                         $logger
     * @param \Magento\Framework\Module\Manager                            $moduleManager
     * @param \Magento\Framework\Model\ResourceModel\AbstractResource|null $resource
     * @param \Magento\Framework\Data\Collection\AbstractDb|null           $resourceCollection
     * @param array                                                        $data
     */
    public function __construct(
        \Magento\Framework\Model\Context                        $context,
        \Magento\Framework\Registry                             $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory       $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory            $customAttributeFactory,
        \Magento\Payment\Helper\Data                            $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface      $scopeConfig,
        \Magento\Payment\Model\Method\Logger                    $logger,
        \Magento\Framework\Module\Manager                       $moduleManager,
        \Revolut\Payment\Helper\ConfigHelper                    $configHelper,
        \Revolut\Payment\Model\RevolutOrderFactory              $revolutOrderFactory,
        \Revolut\Payment\Helper\DataHelper                      $dataHelper,
        \Magento\Framework\Json\Helper\Data                     $jsonHelper,
        \Magento\Framework\Message\ManagerInterface             $managerInterface,
        \Revolut\Payment\Helper\Logger                          $loggerRevolut,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb           $resourceCollection = null,
        array                                                   $data = []
    ) {
        $this->moduleManager = $moduleManager;
        $this->configHelper = $configHelper;
        $this->revolutOrderFactory = $revolutOrderFactory;
        $this->dataHelper = $dataHelper;
        $this->jsonHelper = $jsonHelper;
        $this->managerInterface = $managerInterface;
        $this->loggerRevolut = $loggerRevolut;
        parent::__construct($context, $registry, $extensionFactory, $customAttributeFactory, $paymentData, $scopeConfig, $logger, $resource, $resourceCollection, $data);
    }

    /**
     * @return \Magento\Payment\Model\Method\AbstractMethod|RevolutForm
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function validate()
    {
        return \Magento\Payment\Model\Method\AbstractMethod::validate();
    }

    /**
     * @param  \Magento\Quote\Api\Data\CartInterface|null $quote
     * @return bool
     */
    public function isAvailable(\Magento\Quote\Api\Data\CartInterface $quote = null)
    {
        if (!$this->moduleManager->isEnabled(ConstantValue::MODULE_NAME)) {
            return false;
        }
        return \Magento\Payment\Model\Method\AbstractMethod::isAvailable($quote);
    }

    /**
     * @param  \Magento\Framework\DataObject $data
     * @return $this|RevolutForm
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function assignData(\Magento\Framework\DataObject $data)
    {
        $infoInstance = $this->getInfoInstance();
        $additionalData = $data->getData('additional_data');

        if (!is_array($additionalData)) {
            return $this;
        }

        if (!isset($additionalData['publicId']) || empty($additionalData['publicId'])) {
            return $this;
        }

        //get revolut order from API
        $publicId = $additionalData['publicId'];
        $revolutOrderFactory = $this->revolutOrderFactory->create();
        $revolutOrder = $revolutOrderFactory->getRevolutOrder($publicId, 'public_id');
        // @phpstan-ignore-next-line
        $revolutOrderId = (string) $revolutOrder->getData('revolut_order_id');

        if (empty($revolutOrderId)) {
            throw new \Magento\Framework\Exception\LocalizedException(__('Could not load the Order publicId: ' . $publicId));
        }

        $revolutOrder = $this->dataHelper->getRevolutOrder($revolutOrderId);
        $revolutOrderJsonData = $this->jsonHelper->jsonEncode($revolutOrder);

        if (!isset($revolutOrder['id'])) {
            throw new \Magento\Framework\Exception\LocalizedException(__('Can not retrieve the Order publicId: '  . $publicId));
        }

        //log detail payment
        $this->loggerRevolut->debug("assignData  - " . $revolutOrder['id'] . ' - ' . $revolutOrder['state']);


        $infoInstance->setAdditionalInformation('publicId', $publicId);
        $infoInstance->setAdditionalInformation('revolutOrderData', $revolutOrderJsonData);
        $infoInstance->setAdditionalInformation('revolutOrderId', $revolutOrderId);
        return $this;
    }

    /**
     * Get config payment action url
     * Used to universalize payment actions when processing payment place
     */
    public function getConfigPaymentAction()
    {
        try {
            $info = $this->getInfoInstance();
            $order = $info->getOrder();
            $id_quote = $order->getQuoteId();
            $revolutOrderFactory = $this->revolutOrderFactory->create();
            $revolutOrder = $revolutOrderFactory->getRevolutOrder($id_quote, 'quote_id');
            $revolutOrderId = $revolutOrder->getData('revolut_order_id');
    
            if (empty($revolutOrderId)) {
                throw new \Magento\Framework\Exception\LocalizedException(__('Could not find the Order'));
            }
    
            $this->loggerRevolut->debug("getConfigPaymentAction - " . $id_quote . " - " . $revolutOrderId);
    
            $revolutOrder = $this->dataHelper->getRevolutOrder($revolutOrderId);
    
            if (!isset($revolutOrder['id'])) {
                throw new \Magento\Framework\Exception\LocalizedException(__("Can not retrieve the Order: " . $id_quote . " - " . $revolutOrderId));
            }
            
            if ($this->configHelper->isManualCaptureEvent()) {
                $this->loggerRevolut->debug("do not capture authorize only order:"  . $id_quote . " - " . $revolutOrderId);
                return ConstantValue::MAGENTO_AUTHORIZE;
            }
            
            $this->loggerRevolut->debug("capture order automatically: " . $id_quote . " - " . $revolutOrderId);
            $this->captureRevolutOrder($revolutOrderId);
    
            for ($i = 0; $i <= 9; $i++) {
                $revolutOrder = $this->dataHelper->getRevolutOrder($revolutOrder['id']);
                if (!empty($revolutOrder['state'])) {
                    if ($revolutOrder['state'] == 'COMPLETED' || $revolutOrder['state'] == "IN_SETTLEMENT") {
                        return ConstantValue::MAGENTO_AUTHORIZE_CAPTURE;
                    }
                }
    
                sleep(3);
            }
    
            return ConstantValue::MAGENTO_AUTHORIZE;
        } catch (\Exception $e) {
            // @phpstan-ignore-next-line
            $this->loggerRevolut->debug("getConfigPaymentAction - error: " . $revolutOrderId . " - " . $e->getMessage());
        }

        return ConstantValue::MAGENTO_AUTHORIZE;
    }

    /**
     * Flag if we need to run payment initialize while order place
     *
     * @return bool
     * @api
     */
    public function isInitializeNeeded()
    {
        $payment_action = $this->configHelper->getConfigValue(ConfigHelper::PAYMENT_ACTION);
        return $payment_action == ConstantValue::MAGENTO_AUTHORIZE;
    }

    /**
     * Method that will be executed instead of authorize or capture
     * if flag isInitializeNeeded set to true
     *
     * @param string $paymentAction
     * @param object $stateObject
     *
     * @return                                        $this
     * @api
     */
    public function initialize($paymentAction, $stateObject)
    {
        $info = $this->getInfoInstance();
        $order = $info->getOrder();
        $payment = $order->getPayment();
        $amount = $order->getGrandTotal();

        $this->authorize($payment, $amount);
        return $this;
    }

    /**
     * @param  \Magento\Payment\Model\InfoInterface $payment
     * @param  float                                $amount
     * @return \Revolut\Payment\Model\RevolutAbstractPaymentMethod
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function capture(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $store_id = $payment->getMethodInstance()->getStore();

        if ($payment->getAdditionalInformation('webhook_event_order_completed')) {
            return parent::capture($payment, $amount);
        }

        $revolutOrderId = $payment->getAdditionalInformation('revolutOrderId');
        $revolutOrder = $this->dataHelper->getRevolutOrder($revolutOrderId, $store_id);
        $this->loggerRevolut->debug("start capture action - " . $store_id . " - " . $revolutOrderId);

        // check fraud order
        if ($this->checkFraudOrder($revolutOrder, $payment)) {
            $payment->setIsFraudDetected(true);
        }

        if ($this->configHelper->isManualCaptureEvent($store_id)) {
            if ($this->checkStateRevolutOrder($revolutOrderId, $store_id)) {
                $amount = $this->dataHelper->createRevolutAmount($amount, $revolutOrder['order_amount']['currency']);
                
                $revolutCaptureResult = $this->captureRevolutOrder($revolutOrderId, $amount, $store_id);
                $orderCurrency = $revolutCaptureResult['order_amount']['currency'];
                $this->managerInterface->addSuccess(__('Payment ' . floatval($revolutCaptureResult['order_amount']['value']) / 100 . ' ' . $orderCurrency . ' was successfully captured by Revolut.'));
                $this->loggerRevolut->debug("manual order captured successfully - " . $revolutOrderId . " - " . $payment->getOrder()->getIncrementId());
            }
        }

        $payment->setIsTransactionPending(false);
        $payment->setAdditionalInformation('IsTransactionPending', null);
        $payment->setTransactionId($revolutOrderId);
        $payment->setShouldCloseParentTransaction(true);
        $payment->setIsTransactionClosed(true);
        $payment->setAmount($amount);
        //update revolut order id on Magento DB and Magento Order ID on Revolut API
        $this->setOrderIds($revolutOrderId, $revolutOrder['public_id'], $payment->getOrder()->getIncrementId(), $store_id);
        if (!$this->configHelper->isManualCaptureEvent($store_id)) {
            $this->loggerRevolut->debug("automatic order captured successfully - " . $revolutOrderId . " - " .  $payment->getOrder()->getIncrementId());
        }
            
        return parent::capture($payment, $amount);
    }

    /**
     * @param  \Magento\Payment\Model\InfoInterface $payment
     * @param  float                                $amount
     * @return \Revolut\Payment\Model\RevolutAbstractPaymentMethod
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function authorize(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $store_id = $payment->getMethodInstance()->getStore();

        $revolutOrderId = $payment->getAdditionalInformation('revolutOrderId');
        $revolutOrder = $this->dataHelper->getRevolutOrder($revolutOrderId, $store_id);
        $this->loggerRevolut->debug("start authorize action - " . $store_id . " - " . $revolutOrderId);

        // check fraud order
        if ($this->checkFraudOrder($revolutOrder, $payment)) {
            $payment->setIsFraudDetected(true);
        }

        //update revolut order id on Magento DB and Magento Order ID on Revolut API
        $this->setOrderIds($revolutOrder['id'], $revolutOrder['public_id'], $payment->getOrder()->getIncrementId(), $store_id);

        $payment->setIsTransactionPending(true);
        $payment->setAmount($amount);

        if (!$this->configHelper->isManualCaptureEvent($store_id)) {
            $payment->getOrder()->addStatusHistoryComment(
                'Payment is taking a bit longer than expected to be completed.
							                If the order is not moved to the "Completed order" state after 24h, please check your Revolut account to verify that this payment was taken.
							                You might need to contact your customer if it wasn’t. (Revolut Order ID: ' . $revolutOrder['id'] . ')',
                false
            );
        }

        return parent::authorize($payment, $amount);
    }

    /**
     * @param  array                                $revolutOrder
     * @param  \Magento\Payment\Model\InfoInterface $payment
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function checkFraudOrder($revolutOrder, $payment)
    {
        if (empty($revolutOrder['id']) || empty($revolutOrder['state']) || in_array($revolutOrder['state'], array('CANCELLED', 'FAILED', 'PENDING'))) {
            $this->loggerRevolut->debug("checkFraudOrder: " . \json_encode($revolutOrder));
            throw new \Magento\Framework\Exception\LocalizedException(__('An error occured while payment process'));
        }

        $currency = $revolutOrder['order_amount']['currency'];
        $revolut_amount = $revolutOrder['order_amount']['value'];
        $calculated_revolut_amount = $this->dataHelper->createRevolutAmount($payment->getAmountOrdered(), $currency);

        if (abs($calculated_revolut_amount - $revolut_amount) > 50) {
            $this->loggerRevolut->debug("Revolut Authorize Amount Not Same Order Amount - " . $revolutOrder['id'] . "-" . $revolut_amount . " - " . $calculated_revolut_amount . " - " . $payment->getAmountOrdered() . " - " . $currency);
            return true;
        }

        return false;
    }

    /**
     * @param  \Magento\Payment\Model\InfoInterface $payment
     * @param  float                                $amount
     * @return \Revolut\Payment\Model\RevolutAbstractPaymentMethod
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function refund(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $store_id = $payment->getMethodInstance()->getStore();
        $revolutOrderId = $payment->getAdditionalInformation('revolutOrderId');
        $mage_order_id = $payment->getOrder()->getIncrementId();

        $this->loggerRevolut->debug("start refund action - " . $revolutOrderId . " - " . $store_id . " - " . $mage_order_id);

        $revolutOrder = $this->dataHelper->getRevolutOrder($revolutOrderId, $store_id);

        if (!isset($revolutOrder['id'])) {
            $this->loggerRevolut->debug("Refund error: Revolut Order does not exist: " . $revolutOrderId  . " - " . json_encode($revolutOrder));
            throw new \Magento\Framework\Exception\LocalizedException(__('Revolut Order does not exist'));
        }

        //check order already REFUND
        $amountRefunded = $revolutOrder['refunded_amount']['value'];
        $orderCurrency = $revolutOrder['order_amount']['currency'];

        if ($amountRefunded && $amountRefunded == $revolutOrder['order_amount']['value']) {
            $this->managerInterface->addSuccess(__('Refund amount: ' . $amount . $orderCurrency . ' was already exist.'));
            return parent::refund($payment, $amount);
        }

        
        $response = $this->dataHelper->refundRevolutOrder($revolutOrderId, $amount, $orderCurrency, $mage_order_id, $store_id);
        $responseJson = $this->jsonHelper->jsonEncode($response);
        //log detail payment
        $this->loggerRevolut->debug("refund revolut order - " . $revolutOrderId . " - " . $mage_order_id . " - " . $responseJson);
        //handle error
        if (!isset($response['id'])) {
            throw new \Magento\Framework\Exception\LocalizedException(__('Cannot Refund Order - Error Id: ' . $response['errorId']));
        }

        $this->managerInterface->addSuccess(__($amount . ' ' . $orderCurrency . ' successfully refunded by Revolut.'));
        return parent::refund($payment, $amount);
    }

    /**
     * @param  \Magento\Payment\Model\InfoInterface $payment
     * @return \Revolut\Payment\Model\RevolutAbstractPaymentMethod
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function void(\Magento\Payment\Model\InfoInterface $payment)
    {
        $revolutOrderId = $payment->getAdditionalInformation('revolutOrderId');
        $store_id = $payment->getMethodInstance()->getStore();

        $this->loggerRevolut->debug("start cancel action - " . $store_id . ' - ' . $revolutOrderId);

        $checkState = $this->checkStateRevolutOrder($revolutOrderId, $store_id);

        if (!$checkState) {
            throw new \Magento\Framework\Exception\LocalizedException(__('Revolut: Cannot Cancel Order - ' . $revolutOrderId));
        }
        
        $response = $this->dataHelper->cancelRevolutOrder($revolutOrderId, $store_id);
        $responseJson = $this->jsonHelper->jsonEncode($response);
        //log detail payment
        $this->loggerRevolut->debug("cancel revolut order - " . $revolutOrderId . " - " . $payment->getOrder()->getIncrementId() . " - " . $responseJson);

        //handle error
        if (!isset($response['id'])) {
            throw new \Magento\Framework\Exception\LocalizedException(__('Cannot Cancel Order - Error Id: ' . $response['errorId']));
        }

        $this->managerInterface->addSuccess(__('Cancel order by Revolut success.'));
        return parent::void($payment);
    }

    /**
     * @param  string|mixed  $revolutOrderId
     * @param  float|boolean $amount
     * @param  int|boolean   $store_id
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function captureRevolutOrder($revolutOrderId, $amount = false, $store_id = false)
    {
        $revolutOrder = $this->dataHelper->getRevolutOrder($revolutOrderId);

        if (empty($revolutOrder['id']) || empty($revolutOrder['state'])) {
            throw new \Magento\Framework\Exception\LocalizedException(__('Capture event can not load order details - ' . $revolutOrderId . " - " .  $this->jsonHelper->jsonEncode($revolutOrder)));
        }

        //check is order already captured
        if ($revolutOrder['state'] == ConstantValue::ORDER_COMPLETED) {
            $this->loggerRevolut->debug("Capture event order already captured - " . $revolutOrderId . " - " . $revolutOrder['state']);
            return $revolutOrder;
        }

        $revolutCaptureResult = $this->dataHelper->captureRevolutOrder($revolutOrderId, $amount, $store_id);
        //log detail payment
        $this->loggerRevolut->debug("Capture event - " . $revolutOrderId . " - " . $revolutCaptureResult['state']);

        //handle error
        if (!isset($revolutCaptureResult['id'])) {
            throw new \Magento\Framework\Exception\LocalizedException(__('Cannot Capture Order - Error Id: ' . $revolutCaptureResult['errorId']));
        }

        return $revolutCaptureResult;
    }

    /**
     * @param  string|mixed $revolutOrderId
     * @param  int|bool     $store_id
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function checkStateRevolutOrder($revolutOrderId, $store_id = false)
    {
        //check state revolut order
        $revolutOrder = $this->dataHelper->getRevolutOrder($revolutOrderId, $store_id);
        if (!isset($revolutOrder['id'])) {
            return false;
        }

        //check order already COMPLETED or CANCEL
        if ($revolutOrder['state'] == ConstantValue::ORDER_COMPLETED || $revolutOrder['state'] == ConstantValue::ORDER_CANCELLED) {
            return false;
        }

        return true;
    }

    /**
     * @param  string|mixed $revolutOrderId
     * @param  string|mixed $revolutOrderPublicId
     * @param  int|mixed    $magentoOrderId
     * @param  int|bool     $store_id
     * @return void
     */
    private function setOrderIds($revolutOrderId, $revolutOrderPublicId, $magentoOrderId, $store_id)
    {
        $revolutOrderFactory = $this->revolutOrderFactory->create();
        $revolutOrderModel = $revolutOrderFactory->getRevolutOrder($revolutOrderPublicId, 'public_id');
        $revolutOrderFactory->updateRevolutOrderField($revolutOrderModel, 'increment_order_id', $magentoOrderId);
        $this->dataHelper->updateRevolutOrderMerchantRef($revolutOrderId, $magentoOrderId, $store_id);
    }
}
