<?php

namespace Revolut\Payment\Model\Api;

use http\Env\Request;
use Revolut\Payment\Api\RevolutOrderApiServiceInterface;
use Revolut\Payment\Helper\Logger;
use Revolut\Payment\Model\ConstantValue;
use Revolut\Payment\Helper\ConfigHelper;
use Revolut\Payment\Model\RevolutOrderFactory;
use Magento\Framework\Exception\LocalizedException;

/**
 * Class RevolutOrderApiService
 *
 * @package Revolut\Payment\Model\Api
 */
class RevolutOrderApiService implements RevolutOrderApiServiceInterface
{
    /**
     * @var \Revolut\Payment\Helper\DataHelper
     */
    protected $dataHelper;

    /**
     * @var \Magento\Framework\Json\Helper\Data
     */
    protected $jsonHelper;

    /**
     * @var ConfigHelper
     */
    protected $configHelper;

    /**
     * @var \Magento\Checkout\Model\Session
     */
    protected $checkoutSession;

    /**
     * @var \Magento\Catalog\Model\ProductFactory
     */
    protected $productFactory;

    /**
     * @var \Magento\Framework\Webapi\Rest\Request
     */
    protected $request;

    /**
     * @var \Revolut\Payment\Model\RevolutOrderFactory
     */
    protected $revolutOrderFactory;

    /**
     * @var \Revolut\Payment\Model\ResourceModel\RevolutOrder
     */
    protected $revolutOrderResourceModel;

    /**
     * @var String
     */
    protected $supportLink;

    /**
     * @var Logger
     */
    protected $loggerRevolut;

    /**
     * @var \Magento\Checkout\Model\Cart
     */
    private $cart;

    public function __construct(
        \Revolut\Payment\Helper\DataHelper         $dataHelper,
        \Magento\Framework\Json\Helper\Data        $jsonHelper,
        \Revolut\Payment\Helper\ConfigHelper       $configHelper,
        \Magento\Checkout\Model\Session            $checkoutSession,
        \Magento\Catalog\Model\ProductFactory      $productFactory,
        \Revolut\Payment\Model\ResourceModel\RevolutOrder $revolutOrderResourceModel,
        \Magento\Framework\Webapi\Rest\Request     $request,
        \Revolut\Payment\Helper\Logger             $loggerRevolut,
        \Revolut\Payment\Model\RevolutOrderFactory $revolutOrderFactory,
        \Magento\Checkout\Model\Cart               $cart
    ) {
        $this->dataHelper = $dataHelper;
        $this->jsonHelper = $jsonHelper;
        $this->configHelper = $configHelper;
        $this->checkoutSession = $checkoutSession;
        $this->productFactory = $productFactory;
        $this->request = $request;
        $this->revolutOrderFactory = $revolutOrderFactory;
        $this->revolutOrderResourceModel = $revolutOrderResourceModel;
        $this->loggerRevolut = $loggerRevolut;
        $this->cart = $cart;
        $this->supportLink = 'https://www.revolut.com/en-HR/business/help/merchant-accounts/';
        $this->supportLink .= 'payments/in-which-currencies-can-i-accept-payments';
    }

    /**
     * Get or Create Magento Quote
     *
     * @return \Magento\Quote\Model\Quote
     */
    public function getOrCreateQuote()
    {
        $quote = $this->checkoutSession->getQuote();
        if (!$quote->getId()) {
            $quote = $this->cart->getQuote();
            $quote->setIsActive(true)->save();
        }

        return $quote;
    }

     /**
      * @param  int $productId
      * @return \Magento\Catalog\Model\Product
      */
    public function getProduct($productId)
    {
        return $this->productFactory->create()->load($productId);
    }

     /**
      * @param  int|mixed $quoteId
      * @param  string    $currency
      * @param  float     $amount
      * @return array
      */
    public function createRevolutOrder($quoteId, $currency, $amount)
    {
        if (!$this->configHelper->checkCurrencySupport($currency)) {
            throw new LocalizedException(__($currency . " currency is not supported, please use a different currency to checkout. You can check the supported currencies from the link: " . ConstantValue::SUPPORT_LINK));
        }

        $revolutOrderFactory = $this->revolutOrderFactory->create();
        $revolutOrder = $revolutOrderFactory->getRevolutOrder($quoteId, 'quote_id');
        $revolutOrderId = $revolutOrder->getData('revolut_order_id');
            
        $this->loggerRevolut->debug("start create order revolutOrderId: " . $revolutOrderId . " quoteId: " . $quoteId);


        if (!empty($revolutOrderId)) {
            $revolutOrderData = $this->dataHelper->getRevolutOrder($revolutOrderId);
            if (!empty($revolutOrderData['id']) && $revolutOrderData['state'] == 'PENDING') {
                $orderData = $this->dataHelper->updateRevolutOrderAmount($revolutOrderId, $amount, $currency);
                if (!isset($orderData['public_id'])) {
                    $this->loggerRevolut->debug("API update Order failed: " . $this->jsonHelper->jsonEncode($orderData));
                    throw new LocalizedException(__('An error occurred while processing the order'));
                }

                //update revolut order amount on Magento DB
                $revolutOrderFactory->updateRevolutOrderField($revolutOrder, 'order_amount', $orderData['order_amount']['value']);
                $this->loggerRevolut->debug("existing order updated successfully: " . $orderData['id'] . ' - ' . $orderData['public_id']);
                return [
                    'success' => true,
                    'public_id' => $orderData['public_id'],
                    'currency' => $currency,
                    'amount' => $this->dataHelper->createRevolutAmount($amount, $currency),
                    'available_card_brands' => $this->dataHelper->getAvailableCardBrands($orderData['public_id']),
                    'public_key' => $this->dataHelper->getMerchantPublicKey(),
                ];
            } else {
                //if there is a problem remove current order in order to create a new one from scratch
                $revolutOrder->delete();
            }

            if (isset($revolutOrderData['state']) && $revolutOrderData['state'] == ConstantValue::ORDER_AUTHORISED) {
                // cancel order if it has been paid
                $this->loggerRevolut->debug("start canceling current order during order create event - " . $revolutOrderId . " - quoteId " . $quoteId);
                $canceledOrderData = $this->dataHelper->cancelRevolutOrder($revolutOrderId);
               
                if (!isset($canceledOrderData['id'])) {
                    $this->loggerRevolut->debug('during order create event canceling current order process has been failed - ' . $revolutOrderId . " - quoteId " . $quoteId . ' - response: ' . $this->jsonHelper->jsonEncode($canceledOrderData));
                }
               
                $this->loggerRevolut->debug("canceling current order during order create event has been successfully completed - " . $revolutOrderId . " - quoteId " . $quoteId);
            }
        }

        $orderData = $this->dataHelper->createRevolutOrder($amount, $currency);

        //handle create revolut order error
        if (empty($orderData['id'])) {
            $this->loggerRevolut->debug("API create Order failed: " . $this->jsonHelper->jsonEncode($orderData));
            throw new LocalizedException(__('An error occurred while processing the order'));
        }

        $orderData['quote_id'] = $quoteId;
        //save revolut order
        $revolutOrder = $this->revolutOrderFactory->create();
        $revolutOrder->saveRevolutOrder($orderData);
        $this->loggerRevolut->debug("order created successfully revolutOrderId: " . $orderData['id'] . " quoteId: " . $quoteId);

        return [
            'success' => true,
            'public_id' => $orderData['public_id'],
            'currency' => $currency,
            'amount' => $this->dataHelper->createRevolutAmount($amount, $currency),
            'available_card_brands' => $this->dataHelper->getAvailableCardBrands($orderData['public_id']),
            'public_key' => $this->dataHelper->getMerchantPublicKey(),
        ];
    }

    /**
     * Returns public_id form revolut
     *
     * @return string public_id form revolut.
     * @api
     */
    public function createOrder()
    {
        try {
            $quote = $this->checkoutSession->getQuote();
            $quoteId = $quote->getId();

            if (empty($quoteId)) {
                $quote = $this->getOrCreateQuote();
                $quoteId = $quote->getId();
            }

            $this->loggerRevolut->debug('createOrder quoteId: ' . $quoteId);

            if (empty($quoteId)) {
                throw new LocalizedException(__('Can not load shopping cart'));
            }

            $amount = $quote->getGrandTotal();
            $currency = $quote->getQuoteCurrencyCode();
            $result = $this->createRevolutOrder($quoteId, $currency, $amount);
        } catch (\Exception $e) {
            $this->loggerRevolut->debug('An error occurred while processing the order: ' . $e->getMessage());
            $result = [
                'error' => true,
                'mess' => 'An error occurred while processing the order' . $e->getMessage()
            ];
        }

        return $this->jsonHelper->jsonEncode($result);
    }

    /**
     * Updates total amount on revolut
     *
     * @return String.
     * @api
     */
    public function updateOrder()
    {
        try {
            $params = $this->request->getBodyParams();
            $revolutPublicId = $params['revolut_public_id'];

            $revolutOrderFactory = $this->revolutOrderFactory->create();
            $revolutOrder = $revolutOrderFactory->getRevolutOrder($revolutPublicId, 'public_id');

            $revolutOrderId = $revolutOrder->getData('revolut_order_id');
            
            $mage_increment_order_id = $revolutOrder->getData('increment_order_id');

            if (!empty($mage_increment_order_id)) {
                throw new LocalizedException(__('Magento Order already created for revolutOrderId: ' . $revolutOrderId));
            }

            if (empty($revolutOrderId)) {
                throw new LocalizedException(__('Could not find Revolut Order ID revolut_public_id: ' . $revolutPublicId));
            }

            $quote = $this->checkoutSession->getQuote();
            $quoteId = $quote->getId();
            $amount = $quote->getGrandTotal();
            $currency = $quote->getQuoteCurrencyCode();

            // update revolut order amount on Revolut Api
            $orderData = $this->dataHelper->updateRevolutOrderAmount($revolutOrderId, $amount, $currency);

            if (!isset($orderData['public_id'])) {
                throw new LocalizedException(__("Can not update Revolut Order revolutOrderId: " . $revolutOrderId . " quoteId: " . $quoteId));
            }

            //update revolut order amount on Magento DB
            $revolutOrderFactory->updateRevolutOrderField($revolutOrder, 'order_amount', $orderData['order_amount']['value']);
            $this->loggerRevolut->debug("order updated successfully  revolutOrderId: " . $revolutOrderId . " quoteId: " . $quoteId);

            $result = [
                'success' => true,
                'public_id' => $orderData['public_id'],
                'currency' => $currency,
                'amount' => $this->dataHelper->createRevolutAmount($amount, $currency),
                'public_key' => $this->dataHelper->getMerchantPublicKey(),
            ];
        } catch (\Exception $e) {
            $this->loggerRevolut->debug($e->getMessage());
            $result = [
                'success' => false,
                'mess' => 'An error occurred while processing the order'
            ];
        }

        return $this->jsonHelper->jsonEncode($result);
    }

    /**
     * Cancel order on revolut
     *
     * @return String.
     * @api
     */
    public function cancelOrder()
    {
        try {
            $params = $this->request->getBodyParams();
            $revolutPublicId = $params['revolut_public_id'];
            $this->loggerRevolut->debug("order cancel action requested - " . $revolutPublicId);

            $revolutOrderFactory = $this->revolutOrderFactory->create();
            $revolutOrder = $revolutOrderFactory->getRevolutOrder($revolutPublicId, 'public_id');

            $revolutOrderId = $revolutOrder->getData('revolut_order_id');
            $quoteId = $revolutOrder->getData('quote_id');
            $this->loggerRevolut->debug("order cancel action started - " . $revolutOrderId . " - " . $quoteId);


            if (empty($revolutOrderId)) {
                throw new \Magento\Framework\Exception\LocalizedException(__('Could not find Revolut Order ID revolut_public_id: ' . $revolutPublicId));
            }

            // cancel order on Revolut Api
            $orderData = $this->dataHelper->cancelRevolutOrder($revolutOrderId);

            if (!isset($orderData['id'])) {
                throw new \Magento\Framework\Exception\LocalizedException(__('Can not cancel Revolut Order revolutOrderId: ' . $revolutOrderId . ' response: ' . $this->jsonHelper->jsonEncode($orderData)));
            }

            $revolutOrder->delete();
            $this->loggerRevolut->debug("order canceled successfully revolutOrderId: " . $revolutOrderId . " quoteId: " . $quoteId);
            $result = [
                'success' => true,
            ];
        } catch (\Exception $e) {
            $this->loggerRevolut->debug($e->getMessage());
            $result = [
                'success' => false,
                'mess' => 'An error occurred while canceling the order'
            ];
        }

        return $this->jsonHelper->jsonEncode($result);
    }
    
    /**
     * Register Cashback offer for customer
     *
     * @return bool|string.
     * @api
     */
    public function registerCashback()
    {
        try {
            $params = $this->request->getBodyParams();
            $orderId = $params['order_id'];
            
            if (empty($orderId)) {
                return false;
            }

            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
            // @phpstan-ignore-next-line
            $order = $objectManager->create('\Magento\Sales\Model\Order')->load($orderId);

            $incrementOrderId = $order->getIncrementId();
            $revolutOrderModel = $this->revolutOrderFactory->create();
            $this->revolutOrderResourceModel->load($revolutOrderModel, $incrementOrderId, 'increment_order_id');
            $revolutOrderId = $revolutOrderModel->getData('revolut_order_id');
    
            $paymentToken = (string) $this->dataHelper->getPaymentToken($revolutOrderId);
            // @phpstan-ignore-next-line
            $phone = $order->getBillingAddress()->getTelephone();

            $this->loggerRevolut->debug("cashback register requested - " . $revolutOrderId . " - " . $phone . " - " . $paymentToken);
            $this->dataHelper->registerCashbackCandidate($phone, $paymentToken);
            
            $result = [
                'success' => true,
            ];
        } catch (\Exception $e) {
            $this->loggerRevolut->debug($e->getMessage());
            $result = [
                'success' => false,
                'mess' => 'An error occurred while canceling the order'
            ];
        }

        return $this->jsonHelper->jsonEncode($result);
    }
}
