<?php

namespace Revolut\Payment\Model\Api;

use http\Env\Request;
use Revolut\Payment\Api\PaymentRequestButtonServiceInterface;
use Revolut\Payment\Helper\Logger;
use Revolut\Payment\Helper\ConfigHelper;
use Magento\Framework\Exception\LocalizedException;

/**
 * Class CreateOrder
 *
 * @package Revolut\Payment\Model\Api
 */
class PaymentRequestButtonService implements PaymentRequestButtonServiceInterface
{
    /**
     * @var \Revolut\Payment\Helper\DataHelper
     */
    protected $dataHelper;

    /**
     * @var \Magento\Framework\Json\Helper\Data
     */
    protected $jsonHelper;

    /**
     * @var ConfigHelper
     */
    protected $configHelper;

    /**
     * @var \Magento\Checkout\Model\Session
     */
    protected $checkoutSession;

    /**
     * @var \Magento\Catalog\Model\ProductFactory
     */
    protected $productFactory;

    /**
     * @var \Magento\Framework\Webapi\Rest\Request
     */
    protected $request;

    /**
     * @var \Revolut\Payment\Model\RevolutOrderFactory
     */
    protected $revolutOrderFactory;

    /**
     * @var Logger
     */
    protected $loggerRevolut;

    /**
     * @var \Magento\Tax\Helper\Data
     */
    private $taxHelper;
    
    /**
     * @var \Magento\Quote\Api\ShipmentEstimationInterface
     */
    private $shipmentEstimation;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var \Magento\Quote\Api\CartRepositoryInterface
     */
    private $quoteRepository;

    /**
     * @var \Magento\Quote\Api\CartManagementInterface
     */
    private $quoteManagement;

    /**
     * @var \Magento\Framework\Event\ManagerInterface
     */
    private $eventManager;

    /**
     * @var \Magento\Checkout\Model\Cart
     */
    private $cart;

    /**
     * @var \Magento\Framework\App\Request\Http
     */
    private $http_request;

    /**
     * @var \Magento\Framework\UrlInterface
     */
    private $urlBuilder;
    
    /**
     * @var \Magento\Directory\Model\CountryFactory
     */
    private $countryFactory;

    /**
     * @var \Magento\Framework\Webapi\ServiceInputProcessor
     */
    private $inputProcessor;

    /**
     * @var \Magento\Customer\Model\Session
     */
    private $customerSession;
    
    /**
     * @var \Magento\Checkout\Api\Data\ShippingInformationInterfaceFactory
     */
    private $shippingInformationFactory;

    /**
     * @var \Magento\Checkout\Api\ShippingInformationManagementInterface
     */
    private $shippingInformationManagement;

    public function __construct(
        \Revolut\Payment\Helper\DataHelper         $dataHelper,
        \Magento\Framework\Json\Helper\Data        $jsonHelper,
        \Revolut\Payment\Helper\ConfigHelper       $configHelper,
        \Magento\Checkout\Model\Session            $checkoutSession,
        \Magento\Catalog\Model\ProductFactory      $productFactory,
        \Magento\Framework\Webapi\Rest\Request     $request,
        \Revolut\Payment\Helper\Logger             $loggerRevolut,
        \Revolut\Payment\Model\RevolutOrderFactory $revolutOrderFactory,
        \Magento\Tax\Helper\Data                   $taxHelper,
        \Magento\Checkout\Model\Cart               $cart,
        \Magento\Framework\App\Request\Http        $http_request,
        \Magento\Framework\UrlInterface            $urlBuilder,
        \Magento\Customer\Model\Session            $customerSession,
        \Magento\Quote\Api\ShipmentEstimationInterface $shipmentEstimation,
        \Magento\Store\Model\StoreManagerInterface  $storeManager,
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository,
        \Magento\Quote\Api\CartRepositoryInterface $quoteRepository,
        \Magento\Quote\Api\CartManagementInterface $quoteManagement,
        \Magento\Framework\Event\ManagerInterface $eventManager,
        \Magento\Directory\Model\CountryFactory $countryFactory,
        \Magento\Framework\Webapi\ServiceInputProcessor $inputProcessor,
        \Magento\Checkout\Api\Data\ShippingInformationInterfaceFactory $shippingInformationFactory,
        \Magento\Checkout\Api\ShippingInformationManagementInterface $shippingInformationManagement
    ) {
        $this->dataHelper = $dataHelper;
        $this->jsonHelper = $jsonHelper;
        $this->configHelper = $configHelper;
        $this->checkoutSession = $checkoutSession;
        $this->productFactory = $productFactory;
        $this->http_request = $http_request;
        $this->urlBuilder = $urlBuilder;
        $this->revolutOrderFactory = $revolutOrderFactory;
        $this->loggerRevolut = $loggerRevolut;
        $this->taxHelper = $taxHelper;
        $this->cart = $cart;
        $this->request = $request;
        $this->customerSession = $customerSession;
        $this->shipmentEstimation = $shipmentEstimation;
        $this->storeManager = $storeManager;
        $this->productRepository = $productRepository;
        $this->quoteRepository = $quoteRepository;
        $this->quoteManagement = $quoteManagement;
        $this->eventManager = $eventManager;
        $this->countryFactory = $countryFactory;
        $this->inputProcessor = $inputProcessor;
        $this->shippingInformationFactory = $shippingInformationFactory;
        $this->shippingInformationManagement = $shippingInformationManagement;
    }

    /**
     * Get or Create Magento Quote
     *
     * @return \Magento\Quote\Model\Quote
     */
    public function getOrCreateQuote()
    {
        $quote = $this->checkoutSession->getQuote();
        if (!$quote->getId()) {
            $quote = $this->cart->getQuote();
            $quote->setIsActive(true)->save();
        }

        return $quote;
    }

    /**
     * @param  int $productId
     * @return \Magento\Catalog\Model\Product
     */
    public function getProduct($productId)
    {
        return $this->productFactory->create()->load($productId);
    }

    /**
     * @return string
     */
    public function addToCart()
    {
        $request = $this->request->getBodyParams();
        $id_carrier = $request['id_carrier'];
        $params = [];
        parse_str($request['request'], $params);

        $productId = (int)$params['product'];
        $related = $params['related_product'];

        $quote = $this->cart->getQuote();

        try {
            $storeId = $this->storeManager->getStore()->getId();
            $product = $this->productRepository->getById($productId, false, $storeId);

            $isUpdated = false;
            foreach ($quote->getAllItems() as $item) {
                if ($item->getProductId() == $productId) {
                    $item = $this->cart->updateItem($item->getId(), $params);
                    if ($item->getHasError()) {
                        throw new LocalizedException(__($item->getMessage()));
                    }

                    $isUpdated = true;
                    break;
                }
            }

            if (!$isUpdated) {
                // @phpstan-ignore-next-line
                $item = $this->cart->addProduct($product, $params);
                if ($item->getHasError()) {
                    throw new LocalizedException(__($item->getMessage()));
                }

                if (!empty($related)) {
                    // @phpstan-ignore-next-line
                    $this->cart->addProductsByIds(explode(',', $related));
                }
            }

            $this->cart->save();

            if ($id_carrier) {
                if (!$quote->isVirtual()) {
                    $quote->getShippingAddress()->setShippingMethod($id_carrier)
                        ->setCollectShippingRates(true)
                        ->collectShippingRates();
                }
            }

            $quote->setTotalsCollectedFlag(false);
            $quote->collectTotals();
            $quote->save();

            $amount = $quote->getGrandTotal();
            $currency = $quote->getQuoteCurrencyCode();

            if (empty($currency)) {
                $currency = $quote->getStore()->getCurrentCurrency()->getCode();
            }
            
            $result = [
            'status' => 'success',
            'success' => true,
            'currency' => $currency,
            'shippingOptions' => array(
               [ "id" => "free_shipping","amount" => "0", "description" => "", "label" =>"-"]
            ),
            'total' => [
                'amount' => $this->dataHelper->createRevolutAmount($amount, $currency),
            ],
                
            ];
        } catch (\Exception $e) {
            $this->loggerRevolut->debug($e->getMessage());
            $result = [
            'status' => 'error',
            'success' => false,
            'message' => $e->getMessage()
            ];
        }

        return $this->jsonHelper->jsonEncode($result);
    }

    /**
     * @return string
     */
    public function loadShippingOptions()
    {
        try {
            $request = $this->request->getBodyParams();
            $this->loggerRevolut->debug(\json_encode($request));
            $revolut_public_id = $request['revolut_public_id'];
            $address = $request['address_data'];
            $quote = $this->cart->getQuote();
            $currency = $quote->getQuoteCurrencyCode();
            $rates = [];

            if (!$quote->isVirtual()) {
                $shippingAddress = $this->reFormatAddress($address);
                $quote->getShippingAddress()->addData($shippingAddress);
                $rates = $this->shipmentEstimation->estimateByExtendedAddress($quote->getId(), $quote->getShippingAddress());
            } else {
                $currency = $quote->getQuoteCurrencyCode();
                $amount = $quote->getGrandTotal();
                $this->loggerRevolut->debug(\json_encode($this->updateRevolutOrder($revolut_public_id, $amount, $currency)));

                $shipping_options = [
                    ["id"=> "free_shipping",
                    "amount"=> "0",
                    "description"=> "",
                    "label"=> "-"
                    ]
                ];
                $result = [
                    'status' => 'success',
                    'success' => true,
                    'currency' => $currency,
                    'shippingOptions' => $shipping_options,
                    'total' => [
                        'amount' => $this->dataHelper->createRevolutAmount($quote->getGrandTotal(), $currency),
                    ],
                ];

                return $this->jsonHelper->jsonEncode($result);
            }

            $shouldInclTax = $this->shouldCartPriceInclTax($quote->getStore());
            $shipping_options = [];
            foreach ($rates as $rate) {
                if ($rate->getErrorMessage()) {
                    continue;
                }
                $rate_amount = $shouldInclTax ? $rate->getPriceInclTax() : $rate->getPriceExclTax();
                $shipping_options[] = [
                'id' => $rate->getCarrierCode() . '_' . $rate->getMethodCode(),
                'label' => implode(' - ', [$rate->getCarrierTitle(), $rate->getMethodTitle()]),
                'detail' => $rate->getMethodTitle(),
                'amount' =>  $this->dataHelper->createRevolutAmount($rate_amount, $currency)
                ];
            }
            
            if (!empty($shipping_options)) {
                $this->setMagentoShipping($shipping_options[0]['id'], $revolut_public_id, $address);
            }

            $currency = $quote->getQuoteCurrencyCode();
            $amount = $this->cart->getQuote()->getGrandTotal();

            $result = [
                'status' => 'success',
                'success' => true,
                'currency' => $currency,
                'shippingOptions' => $shipping_options,
                'total' => [
                    'amount' => $this->dataHelper->createRevolutAmount($this->cart->getQuote()->getGrandTotal(), $currency),
                ],
            ];
        } catch (\Exception $e) {
            $result = [
            'status' => 'error',
            'success' => false,
            'message' => $e->getMessage()
            ];
            $this->loggerRevolut->debug($e->getMessage());
        }

        return $this->jsonHelper->jsonEncode($result);
    }

    /**
     * @return string
     */
    public function setShippingOption()
    {
        $request = $this->request->getBodyParams();
        $this->loggerRevolut->debug(\json_encode($request));
        $shipping_id = $request['id_selected_carrier'];
        $revolut_public_id = $request['revolut_public_id'];
        $address = $request['address_data'];

        return $this->setMagentoShipping($shipping_id, $revolut_public_id, $address);
    }

    /**
     * @param  int    $shipping_id
     * @param  string $revolut_public_id
     * @param  array  $address
     * @return string
     */
    public function setMagentoShipping($shipping_id, $revolut_public_id, $address)
    {
        try {
            $quote = $this->cart->getQuote();

            if (!$quote->isVirtual()) {
                $shippingAddress = $this->reFormatAddress($address);
                $shipping = $quote->getShippingAddress()
                    ->addData($shippingAddress);

                if ($shipping_id) {
                    $shipping->setShippingMethod($shipping_id)
                        ->setCollectShippingRates(true)
                        ->collectShippingRates();

                    $parts = explode('_', $shipping_id);
                    $carrierCode = array_shift($parts);
                    $methodCode = implode("_", $parts);

                    $shippingAddress = $this->inputProcessor->convertValue($shippingAddress, 'Magento\Quote\Api\Data\AddressInterface');

                    $shippingInformation = $this->shippingInformationFactory->create();
                    $shippingInformation->setShippingAddress($shippingAddress)->setShippingCarrierCode($carrierCode)->setShippingMethodCode($methodCode);
                    $this->shippingInformationManagement->saveAddressInformation($quote->getId(), $shippingInformation);

                    $quote->setTotalsCollectedFlag(false);
                    $quote->collectTotals();
                }
            }

            $currency = $quote->getQuoteCurrencyCode();
            $amount = $quote->getGrandTotal();
            $this->updateRevolutOrder($revolut_public_id, $amount, $currency);

            $result = [
                'status' => 'success',
                'success' => true,
                'currency' => $currency,
                'total' => [
                    'amount' => $this->dataHelper->createRevolutAmount($amount, $currency),
                ],
            ];
        } catch (\Exception $e) {
            $result = [
                'status' => 'error',
                'success' => false,
                'message' => $e->getMessage()
            ];
            $this->loggerRevolut->debug($e->getMessage());
        }

        return $this->jsonHelper->jsonEncode($result);
    }

    /**
     * @param  array $address
     * @return array
     */
    public function reFormatAddress($address)
    {
        if (!is_array($address) || empty($address['country'])) {
            throw new LocalizedException(__('invalid address request'));
        }

        if (empty($address['recipient'])) {
            $firstName = null;
            $lastName = null;
        } else {
            list($firstName, $lastName) = $this->parseName($address['recipient']);
        }

        if (empty($address['region'])) {
            $regionId = null;
        } else {
            $regionId = $this->getRegionIdBy($address['region'], $address['country']);
        }

        if (empty($address['postalCode'])) {
            $address['postalCode'] = null;
        }

        if (empty($address['phone'])) {
            $address['phone'] = null;
        }

        return [
            'firstname' => $firstName,
            'lastname' => $lastName,
            'email' =>(empty($address['email']) ? "" : $address['email']),
            'street' => (empty($address['addressLine']) ? null : $address['addressLine']),
            'city' => $address['city'],
            'region_id' => $regionId,
            'region' => (empty($address['region']) ? null : $address['region']),
            'postcode' => $address['postalCode'],
            'country_id' => $address['country'],
            'telephone' => $address['phone'],
            'fax' => ''
        ];
    }

    /**
     * @param  array $address
     * @return array
     */
    public function convertAddresses($address)
    {
        $billingAddress = isset($address['billingAddress']) ? $this->reFormatAddress($address['billingAddress']) : [];
        $shippingAddress = isset($address['shippingAddress']) ? $this->reFormatAddress($address['shippingAddress']) : [];
        
        $billingAddress['email'] = $address['email'];
        $shippingAddress['email'] = $address['email'];

        if (empty($shippingAddress['telephone'])) {
            $shippingAddress['telephone'] = $billingAddress['telephone'];
        }

        return [$billingAddress, $shippingAddress];
    }

     /**
      * @param  string $name
      * @return array
      */
    public function parseName($name)
    {
        if (empty($name) || empty(trim($name))) {
            throw new LocalizedException(__("Invalid name"));
        }

        // @phpstan-ignore-next-line
        $nameParts = explode(' ', preg_replace('!\s+!', ' ', (string)trim($name)));
        // @phpstan-ignore-next-line
        if (count($nameParts) == 0) {
            throw new LocalizedException(__("Invalid name."));
        }

        $firstName = array_shift($nameParts);

        if (empty($firstName)) {
            throw new LocalizedException(__("Invalid name."));
        }

        $lastName = "";

        if ($nameParts) {
            $lastName =  array_shift($nameParts);
        }
        

        if (empty($lastName)) {
            $lastName = "lastname";
        }

        return [$firstName, $lastName];
    }

    /**
     * @param  string $regionName
     * @param  string $regionCountry
     * @return string|null
     */
    public function getRegionIdBy($regionName, $regionCountry)
    {
        $regions = $this->getRegionsForCountry($regionCountry);

        $regionName = strtolower(trim($regionName));

        if (isset($regions['names'][$regionName])) {
            return $regions['names'][$regionName];
        } elseif (isset($regions['codes'][$regionName])) {
            return $regions['codes'][$regionName];
        }

        return null;
    }

    /**
     * @param  string $countryCode
     * @return array
     */
    public function getRegionsForCountry($countryCode)
    {
        $country = $this->countryFactory->create()->loadByCode($countryCode);
        // @phpstan-ignore-next-line
        if (empty($country)) {
            return [];
        }

        $values = array();

        foreach ($country->getRegions() as $region) {
            $values['codes'][trim($region->getCode())] = $region->getId();
            $values['names'][trim($region->getName())] = $region->getId();
        }

        return $values;
    }

    /**
     * @param  int|null $store
     * @return bool
     */
    public function shouldCartPriceInclTax($store = null)
    {
        if ($this->taxHelper->displayCartBothPrices($store)) {
            return true;
        } elseif ($this->taxHelper->displayCartPriceInclTax($store)) {
            return true;
        }

        return false;
    }

    /**
     * @param  string|mixed $revolut_public_id
     * @param  float        $amount
     * @param  string       $currency
     * @return array
     */
    public function updateRevolutOrder($revolut_public_id, $amount, $currency)
    {
        $revolutOrderFactory = $this->revolutOrderFactory->create();
        $revolutOrder = $revolutOrderFactory->getRevolutOrder($revolut_public_id, 'public_id');

        $revolutOrderId = $revolutOrder->getData('revolut_order_id');

        if (empty($revolutOrderId)) {
            throw new LocalizedException(__('Could not find Revolut Order ID'));
        }

        // update revolut order amount on Revolut Api
        $orderData = $this->dataHelper->updateRevolutOrderAmount($revolutOrderId, $amount, $currency);

        if (!isset($orderData['public_id'])) {
            throw new LocalizedException(__('Can not update Revolut Order'));
        }

        //update revolut order amount on Magento DB
        $revolutOrderFactory->updateRevolutOrderField($revolutOrder, 'order_amount', $orderData['order_amount']['value']);
        return $orderData;
    }

    public function validateOrder()
    {
        try {
            $request = $this->request->getBodyParams();
            $this->loggerRevolut->debug((string)\json_encode($request));

            list($billingAddress, $shippingAddress) = $this->convertAddresses($request['address']);

            $id_selected_carrier = $request['id_carrier'];
            $publicId = $request['public_id'];

            $quote = $this->cart->getQuote();
            $quote->setIsWalletButton(true);
        
            $quote->reserveOrderId()->save();
            $quote->getBillingAddress()
                ->addData($billingAddress);

            if (!$quote->isVirtual()) {
                $shipping = $quote->getShippingAddress()
                    ->addData($shippingAddress);
                $shipping->setShippingMethod($id_selected_carrier)
                    ->setCollectShippingRates(true);
            }

            $quote->setTotalsCollectedFlag(false);
            $quote->collectTotals();
            // @phpstan-ignore-next-line
            $this->storeManager->setCurrentStore($quote->getStoreId());

            if (!$this->customerSession->isLoggedIn()) {
                $quote->setCheckoutMethod(\Magento\Checkout\Model\Type\Onepage::METHOD_GUEST)
                    ->setCustomerId(null)
                    ->setCustomerEmail($quote->getBillingAddress()->getEmail())
                    ->setCustomerIsGuest(true)
                    ->setCustomerGroupId(\Magento\Customer\Api\Data\GroupInterface::NOT_LOGGED_IN_ID);
            } else {
                $quote->setCheckoutMethod(\Magento\Checkout\Model\Type\Onepage::METHOD_CUSTOMER);
            }

            $quote->getPayment()->importData(
                ['method' => 'revolut_payment_request_form', 'additional_data' => [
                'publicId' => $publicId
                ]]
            );

            $this->quoteRepository->save($quote);

            $result = [
                'status' => 'success',
                'success' => true,
            ];
        } catch (\Exception $e) {
            $result = [
                'status' => 'error',
                'success' => false,
                'message' => $e->getMessage()
            ];
            $this->loggerRevolut->debug($e->getMessage());
        }

        return $this->jsonHelper->jsonEncode($result);
    }
    
    public function processOrder()
    {
        try {
            $quote = $this->cart->getQuote();
            $order = $this->quoteManagement->submit($quote);
            if ($order) {
                $this->eventManager->dispatch(
                    'checkout_type_onepage_save_order_after',
                    ['order' => $order, 'quote' => $quote]
                );

                $this->checkoutSession
                    ->setLastQuoteId($quote->getId())
                    ->setLastSuccessQuoteId($quote->getId())
                    ->setLastOrderId($order->getId())
                    ->setLastRealOrderId($order->getIncrementId())
                    ->setLastOrderStatus($order->getStatus());
            }

            $this->eventManager->dispatch(
                'checkout_submit_all_after',
                [
                    'order' => $order,
                    'quote' => $quote
                ]
            );

            $result = [
                'status' => 'success',
                'success' => true,
                'redirect_url' => $this->urlBuilder->getUrl('checkout/onepage/success', ['_secure' => $this->http_request->isSecure()])
            ];
        } catch (\Exception $e) {
            $result = [
                'status' => 'error',
                'success' => false,
                'message' => $e->getMessage()
            ];
            $this->loggerRevolut->debug($e->getMessage());
        }

        return $this->jsonHelper->jsonEncode($result);
    }
}
