<?php

namespace Revolut\Payment\Helper;

use Magento\Framework\App\Helper\Context;
use Revolut\Payment\Model\ConstantValue;

/**
 * Class DataHelper
 *
 * @package Revolut\Payment\Helper
 */
class DataHelper extends \Magento\Framework\App\Helper\AbstractHelper
{
    /**
     * @var \Magento\Framework\Module\ModuleListInterface
     */
    protected $moduleList;

    /**
     * @var \Magento\Framework\App\ProductMetadataInterface
     */
    protected $productMetadata;

    /**
     * @var ConfigHelper
     */
    protected $configHelper;
    /**
     * @var \Magento\Framework\Json\Helper\Data
     */
    protected $jsonHelper;

    /**
     * DataHelper constructor.
     *
     * @param Context                                         $context
     * @param \Magento\Framework\Json\Helper\Data             $jsonHelper
     * @param \Magento\Framework\Module\ModuleListInterface   $moduleList
     * @param \Magento\Framework\App\ProductMetadataInterface $productMetadata
     * @param ConfigHelper                                    $configHelper
     */
    public function __construct(
        Context $context,
        \Magento\Framework\Json\Helper\Data $jsonHelper,
        \Magento\Framework\Module\ModuleListInterface $moduleList,
        \Magento\Framework\App\ProductMetadataInterface $productMetadata,
        \Revolut\Payment\Helper\ConfigHelper $configHelper
    ) {
        parent::__construct($context);
        $this->jsonHelper = $jsonHelper;
        $this->moduleList = $moduleList;
        $this->productMetadata = $productMetadata;
        $this->configHelper = $configHelper;
    }

    /**
     * @param  string      $url
     * @param  null|string $content
     * @param  string      $requestMethod
     * @param  bool|int    $store_id
     * @param  string|null $apiKey
     * @param  array $additionalHeaderData
     * @return string
     * @throws \Exception
     */
    public function sendRequest($url, $content = null, $requestMethod = 'POST', $store_id = false, $apiKey = null, $additionalHeaderData = [])
    {
        $apiKey = $apiKey ? $apiKey : $this->configHelper->getApiKey($store_id);
        $httpHeaders = new \Zend\Http\Headers();
        // @phpstan-ignore-next-line
        $setup_version =  $this->moduleList->getOne(ConstantValue::MODULE_NAME)['setup_version'];
        $httpHeaders->addHeaders(
            array_merge(
                [
                "Authorization" => "Bearer " . $apiKey,
                "User-Agent" => 'Revolut Payment Gateway/' . $setup_version
                . ' Magento/' . $this->productMetadata->getVersion()
                . ' PHP/' . PHP_VERSION,
                "Content-Type" => "application/json; charset=utf-8"
                ],
                (array) $additionalHeaderData
            )
        );

        $request = new \Zend\Http\Request();
        $request->setHeaders($httpHeaders);
        $request->setContent($content);
        $request->setUri($url);
        $request->setMethod(strtoupper($requestMethod));


        $client = new \Zend\Http\Client();
        $options = [
            'adapter' => 'Zend\Http\Client\Adapter\Curl',
            'curloptions' => [CURLOPT_FOLLOWLOCATION => true],
            'maxredirects' => 0,
            'timeout' => 30
        ];
        $client->setOptions($options);
        try {
            $response = $client->send($request);
            $responseBody = $response->getBody();
            return $responseBody;
        } catch (\Exception $e) {
            throw new \Exception(__("API error: " . $e->getMessage()));
        }
    }

    /**
     * Create Order via API
     *
     * @param  float  $amount
     * @param  string $currency
     * @return array
     */
    public function createRevolutOrder($amount, $currency)
    {
        $params = [
            "amount" => $this->createRevolutAmount($amount, $currency),
            "currency" => $currency,
            "capture_mode" => ConstantValue::REVOLUT_AUTHORIZE_ONLY
        ];

        if (!$this->configHelper->isManualCaptureEvent()) {
            $params["cancel_authorised_after"] = ConstantValue::AUTO_CANCEL_TIMEOUT;
        }

        $apiUrl = $this->configHelper->getApiUrl() . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_ORDER;
        $response = $this->sendRequest($apiUrl, $this->jsonHelper->jsonEncode($params));
        return (array)$this->jsonHelper->jsonDecode($response);
    }

    /**
     * Update Order amount via API
     *
     * @param  string $revolutOrderId
     * @param  float  $amount
     * @param  string $currency
     * @return array
     */
    public function updateRevolutOrderAmount($revolutOrderId, $amount, $currency)
    {
        $params = [
            "amount" => $this->createRevolutAmount($amount, $currency),
            "currency" => $currency,
        ];

        $apiUrl = $this->configHelper->getApiUrl() . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_ORDER . '/' . $revolutOrderId;

        $response = $this->sendRequest($apiUrl, $this->jsonHelper->jsonEncode($params), 'PATCH');

        return (array)$this->jsonHelper->jsonDecode($response);
    }

    /**
     * Update Order Merchant ref via API
     *
     * @param  string   $revolutOrderId
     * @param  mixed    $merchant_order_id
     * @param  int|bool $store_id
     * @return array
     */
    public function updateRevolutOrderMerchantRef($revolutOrderId, $merchant_order_id, $store_id = false)
    {
        $curlUpdateOrder = $this->configHelper->getApiUrl($store_id) . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_ORDER . '/' . $revolutOrderId;
        $paramOrderUpdate = [
            "merchant_order_id" => $merchant_order_id
        ];

        $response = $this->sendRequest($curlUpdateOrder, $this->jsonHelper->jsonEncode($paramOrderUpdate), 'PATCH', $store_id);
        return (array)$this->jsonHelper->jsonDecode($response);
    }

    /**
     * Refund Order via API
     *
     * @param  string   $revolutOrderId
     * @param  float    $amount
     * @param  string   $currency
     * @param  mixed    $merchant_order_id
     * @param  int|bool $store_id
     * @return array
     */
    public function refundRevolutOrder($revolutOrderId, $amount, $currency, $merchant_order_id, $store_id)
    {
        $curlRefundOrder = $this->configHelper->getApiUrl($store_id) . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_ORDER . '/' . $revolutOrderId . '/' . ConstantValue::ENDPOINT_REFUND_ORDER;
        $paramRefund = [
            "amount" => $this->createRevolutAmount($amount, $currency),
            "merchant_order_ext_ref" => $merchant_order_id,
            "currency" => $currency
        ];
        $response = $this->sendRequest($curlRefundOrder, $this->jsonHelper->jsonEncode($paramRefund), 'POST', $store_id);
        return (array)$this->jsonHelper->jsonDecode($response);
    }
    
    /**
     * @return string
     */
    public function getMerchantPublicKey()
    {
        $publicKeyUrl = $this->configHelper->getMgmtApiUrl() . ConstantValue::API . '/' . ConstantValue::ENDPOINT_PUBLIC_KEY;
        $response = $this->sendRequest($publicKeyUrl, null, "GET");
        
        $publicKey = (array)$this->jsonHelper->jsonDecode($response);
        
        $merchantPublicKey = '';
        $merchantPublicKey = isset($publicKey['public_key']) ? $publicKey['public_key'] : '';
        
        return (string) $merchantPublicKey;
    }
    
    /**
     * @return array
     *
     * @param  string  $publicId
     */
    public function getAvailableCardBrands($publicId)
    {
        $url = $this->configHelper->getMgmtApiUrl() . '/' . ConstantValue::ENDPOINT_ORDER . '/token/' . $publicId;
        $response = $this->sendRequest($url, null, "GET");
        
        $order = (array)$this->jsonHelper->jsonDecode($response);
        
        $availableCardBrands = isset($order['availableCardBrands']) ? $order['availableCardBrands'] : [];
        
        return (array) $availableCardBrands;
    }
    
    /**
     * @return array
     *
     * @param string $paymentToken
     */
    public function getCashbackOffer($paymentToken)
    {
        $url = $this->configHelper->getMgmtApiUrl() . ConstantValue::API . '/public/' . ConstantValue::ENDPOINT_CASHBACK;
        $response = $this->sendRequest($url, null, "GET", false, null, ['Revolut-Payment-Token' => $paymentToken]);
        $cashbackOffer = (array)$this->jsonHelper->jsonDecode($response);

        if (! empty($cashbackOffer) && isset($cashbackOffer['value'])
            && ! empty($cashbackOffer['value']['currency'])
            && ! empty($cashbackOffer['value']['amount'])
        ) {
            $cashbackCurrency = $cashbackOffer['value']['currency'];
            $cashbackAmount   = $cashbackOffer['value']['amount'];
            $cashbackAmount   = $this->convertRevolutAmount($cashbackAmount, $cashbackCurrency);
            
            return array(
                'cashbackCurrency' => $cashbackCurrency,
                'cashbackAmount'   => $cashbackAmount,
            );
        }
        
        return [];
    }
    
    /**
     * @return void
     *
     * @param string $billingPhone
     * @param string $paymentToken
     */
    public function registerCashbackCandidate($billingPhone, $paymentToken)
    {
        $url = $this->configHelper->getMgmtApiUrl() . ConstantValue::API . '/public/' . ConstantValue::ENDPOINT_CASHBACK;
        $response = $this->sendRequest(
            $url,
            $this->jsonHelper->jsonEncode(['phone' => $billingPhone,'marketing_consent' => true,]),
            "POST",
            false,
            null,
            ['Revolut-Payment-Token' => $paymentToken]
        );
    }
    
    /**
     * @return bool|string
     *
     * @param string $revolutOrderId
     */
    public function getPaymentToken($revolutOrderId)
    {
        // @phpstan-ignore-next-line
        $storeId = \Magento\Framework\App\ObjectManager::getInstance()
        ->get(\Magento\Store\Model\StoreManagerInterface::class)
        ->getStore()
        ->getId();

        $revolutOrder = $this->getRevolutOrder($revolutOrderId, $storeId);

        if (! isset($revolutOrder['payments']) || empty($revolutOrder['payments'][0]) || empty($revolutOrder['payments'][0]['token'])) {
            return false;
        }

        return $revolutOrder['payments'][0]['token'];
    }

    /**
     * Cancel Order via API
     *
     * @param  string   $revolutOrderId
     * @param  bool|int $store_id
     * @return array
     */
    public function cancelRevolutOrder($revolutOrderId, $store_id = false)
    {
        $curlCancelOrder = $this->configHelper->getApiUrl($store_id) . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_ORDER . '/' . $revolutOrderId . '/' . ConstantValue::ENDPOINT_CANCEL_ORDER;
        $response = $this->sendRequest($curlCancelOrder, null, "POST", $store_id);
        return (array)$this->jsonHelper->jsonDecode($response);
    }

    /**
     * Get Order via API
     *
     * @param  string   $revolutOrderId
     * @param  bool|int $store_id
     * @return array
     */
    public function getRevolutOrder($revolutOrderId, $store_id = false)
    {
        $curlRetrieveOrder = $this->configHelper->getApiUrl($store_id) . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_ORDER . '/' . $revolutOrderId;
        $revolutOrderData = $this->sendRequest($curlRetrieveOrder, null, 'GET', $store_id);
        return (array)$this->jsonHelper->jsonDecode($revolutOrderData);
    }

    /**
     * Get Order via API
     *
     * @param  string     $revolutOrderId
     * @param  bool|float $amount
     * @param  bool|int   $store_id
     * @return array
     */
    public function captureRevolutOrder($revolutOrderId, $amount, $store_id = false)
    {
        $curlCaptureOrder = $this->configHelper->getApiUrl($store_id) . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_ORDER . '/' . $revolutOrderId . '/' . ConstantValue::ENDPOINT_CAPTURE_ORDER;
        
        if ($amount) {
            $response = $this->sendRequest($curlCaptureOrder, $this->jsonHelper->jsonEncode(["amount" => $amount]), 'POST', $store_id);
        } else {
            $response = $this->sendRequest($curlCaptureOrder, null, 'POST', $store_id);
        }

        return (array)$this->jsonHelper->jsonDecode($response);
    }

    /**
     * @param  string $domain
     * @return string
     */
    public function registerApplePayDomain($domain)
    {
        $curl = $this->configHelper->getApiUrl() . '/api/apple-pay/domains/register';
        return $this->sendRequest($curl, $this->jsonHelper->jsonEncode(["domain" => $domain]));
    }

    /**
     * Check Revolut webhook setup ready
     *
     * @param  string $mageWebHookUrl
     * @param  bool   $sandbox
     * @return bool
     */
    public function checkRevolutWebhookSetup($mageWebHookUrl, $sandbox = false)
    {
        $curl = ConstantValue::URL_PROD . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_WEBHOOK;

        if ($sandbox) {
            $curl = ConstantValue::URL_SANDBOX . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_WEBHOOK;
        }

        $response = $this->sendRequest($curl, null, 'GET');
        $webHookList = (array) $this->jsonHelper->jsonDecode($response);

        if (in_array($mageWebHookUrl, array_column($webHookList, 'url'))) {
            return true;
        }

        return false;
    }
    /**
     * Set via API
     *
     * @param  string $mageWebHookUrl
     * @param  string $apiKey
     * @param  bool   $isTest
     * @return bool
     */

    public function setRevolutWebhook($mageWebHookUrl, $apiKey, $isTest)
    {
        $apiUrl = $isTest ? ConstantValue::URL_SANDBOX : ConstantValue::URL_PROD;
        $curlApiUrl = $apiUrl . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_WEBHOOK;

        $webHookList = $this->getRevolutWebhookList($apiUrl, $apiKey);
        if (in_array($mageWebHookUrl, array_column($webHookList, 'url'))) {
            return true;
        }

        $param = '{"url": "' . $mageWebHookUrl . '","events":["ORDER_COMPLETED"]}';
        $response = $this->sendRequest($curlApiUrl, $param, 'POST', false, $apiKey);
        $response = $this->jsonHelper->jsonDecode($response);
        return isset($response['id']);
    }

    /**
     * Clears Webhook list on API
     *
     * @param  bool $sandbox
     * @return void.
     */
    public function clearRevolutWebhookList($sandbox = true)
    {
        $curl = ConstantValue::URL_PROD . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_WEBHOOK;

        if ($sandbox) {
            $curl = ConstantValue::URL_SANDBOX . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_WEBHOOK;
        }

        $response = $this->sendRequest($curl, null, 'GET');
        $webHookList = $this->jsonHelper->jsonDecode($response);

        foreach ($webHookList as $webhook) {
            $apiUrl = $curl . '/' . $webhook['id'];
            $this->sendRequest($apiUrl, null, 'DELETE');
        }
    }

    /**
     * Retrieves Webhook list on API
     *
     * @param  string $apiUrl
     * @param  string $apiKey
     * @return array
     */
    public function getRevolutWebhookList($apiUrl, $apiKey)
    {
        $curlApiUrl = $apiUrl . ConstantValue::API_VER . '/' . ConstantValue::ENDPOINT_WEBHOOK;
        $response = $this->sendRequest($curlApiUrl, null, 'GET', false, $apiKey);
        return (array)$this->jsonHelper->jsonDecode($response);
    }

    /**
     * Modifies total amount due to API requirements
     *
     * @param  float  $amount
     * @param  string $currency
     * @return float.
     */
    public function createRevolutAmount($amount, $currency)
    {
        $amount = round($amount, 2);
        //YEN currency's minor amount is directly YENs, it does not have fractional values.
        if (!in_array($currency, array('JPY', 'jpy'))) {
            $amount = round($amount * 100);
        }

        return (int)$amount;
    }
    
    /**
     * Modifies total amount due to API requirements
     *
     * @param  float  $amount
     * @param  string $currency
     * @return float.
     */
    public function convertRevolutAmount($amount, $currency)
    {
        //YEN currency's minor amount is directly YENs, it does not have fractional values.
        if (!in_array($currency, array('JPY', 'jpy'))) {
            $amount = round($amount / 100, 2);
        }

        return $amount;
    }
}
