<?php

namespace Revolut\Payment\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Revolut\Payment\Model\ConstantValue;
use Revolut\Payment\Model\Source\Mode;

/**
 * Class ConfigHelper
 *
 * @package Revolut\Payment\Helper
 */
class ConfigHelper extends AbstractHelper
{

    const SANDBOX_MODE = 'payment/revolut_required/mode';
    const APPLE_PAY_DOMAIN_CONFIG = 'payment/revolut_required/apple_pay';
    const HIDE_CASHBACK_BANNER = 'payment/revolut_form/disable_cashback_banner';
    const API_KEY = 'payment/revolut_required/api_key';
    const API_KEY_SANDBOX = 'payment/revolut_required/api_key_sandbox';
    const SHOW_POST_CODE = 'payment/revolut_form/show_post_code';
    const PAYMENT_ACTION = 'payment/revolut_form/payment_action';
    const PR_BUTTONS_CONFIG_KEY = 'payment/revolut_payment_request_form/payment_request_button_';

    /**
     * @var \Magento\Framework\Encryption\EncryptorInterface
     */
    protected $encryptor;

    /**
     * @var \Magento\Config\Model\ResourceModel\Config
     */
    protected $_configInterface;

    /**
     * @var \Magento\Framework\App\Cache\TypeListInterface
     */
    protected $cacheTypeList;


    /**
     * ConfigHelper constructor.
     *
     * @param \Magento\Framework\Encryption\EncryptorInterface $encryptor
     * @param Context                                          $context
     */
    public function __construct(
        \Magento\Framework\Encryption\EncryptorInterface $encryptor,
        \Magento\Config\Model\ResourceModel\Config $configInterface,
        \Magento\Framework\App\Cache\TypeListInterface $cacheTypeList,
        Context $context
    ) {
        $this->encryptor = $encryptor;
        $this->_configInterface = $configInterface;
        $this->cacheTypeList = $cacheTypeList;
        parent::__construct($context);
    }

    /**
     * @param  bool|int $store_id
     * @return bool
     */
    public function isProdMode($store_id = false)
    {
        return $this->getMode($store_id) == Mode::PRODUCTION_MODE;
    }
    
    
    /**
     * @param  mixed $value
     * @return void
     */
    public function setApplePayDomainConfig($value)
    {
        $this->_configInterface->saveConfig(self::APPLE_PAY_DOMAIN_CONFIG, $value, "default", rand());
        $this->cacheTypeList->cleanType(\Magento\Framework\App\Cache\Type\Config::TYPE_IDENTIFIER);
    }

    /**
     * @return bool
     */
    public function checkApplePayDomainRegistration()
    {
        return !$this->isApplePayDomainConfigDone() && $this->isProdMode() && !empty($this->getApiKey());
    }

    /**
     * @return bool
     */
    public function isApplePayDomainConfigDone()
    {
        return $this->scopeConfig->getValue(
            self::APPLE_PAY_DOMAIN_CONFIG,
            "default"
        ) == 'done';
    }
    
    /**
     * @return mixed
     */
    public function hideCashbackBanner()
    {
        return $this->getConfigValue(self::HIDE_CASHBACK_BANNER);
    }
    
    /**
     * @return mixed
     */
    public function getApplePayDomainConfig()
    {
        return $this->scopeConfig->getValue(
            self::APPLE_PAY_DOMAIN_CONFIG,
            "default"
        );
    }

    /**
     * @return array
     */
    public function getPaymentRequestButtonThemeConfigs()
    {
        $size = $this->getConfigValue(self::PR_BUTTONS_CONFIG_KEY. 'size');
        $action = $this->getConfigValue(self::PR_BUTTONS_CONFIG_KEY. 'action');
        $theme = $this->getConfigValue(self::PR_BUTTONS_CONFIG_KEY. 'theme');
        $radius = $this->getConfigValue(self::PR_BUTTONS_CONFIG_KEY. 'radius');
         return [
            'size' => !empty($size) ?  $size  : 'large',
            'action' => !empty($action) ?  $action  : 'buy',
            'theme' => !empty($theme) ?  $theme  : 'dark',
            'radius' => !empty($radius) ?  $radius  : 'none',
         ];
    }

    /**
     * @param  int|bool $store_id
     * @return mixed
     */
    public function getMode($store_id = false)
    {
        return $this->getConfigValue(
            self::SANDBOX_MODE,
            $store_id
        );
    }
    
     /**
      * @param  string $location
      * @return bool
      */
    public function isPaymentRequestButtonAvailableFor($location)
    {
        $locations = (string) $this->getConfigValue(self::PR_BUTTONS_CONFIG_KEY. 'locations');
        $revolut_payment_request_is_active = (bool) $this->getConfigValue('payment/revolut_payment_request_form/active');
        // @phpstan-ignore-next-line
        return $this->isProdMode() && $revolut_payment_request_is_active && in_array($location, explode(',', $locations)) && $this->isProdMode() && !empty($this->getApiKey());
    }
    
     /**
      * @return bool
      */
    public function cardPaymentsIsAvailable()
    {
        $is_active = (bool) $this->getConfigValue('payment/revolut_form/active');
        return $is_active && !empty($this->getApiKey());
    }
    
     /**
      * @return bool
      */
    public function revolutPayIsAvailable()
    {
        $is_active = (bool) $this->getConfigValue('payment/revolut_pay_form/active');
        return $is_active && !empty($this->getApiKey());
    }

    /**
     * @param  int|bool $store_id
     * @return mixed
     */
    public function isManualCaptureEvent($store_id = false)
    {
        return $this->getConfigValue(self::PAYMENT_ACTION, $store_id) != ConstantValue::MAGENTO_AUTHORIZE_CAPTURE;
    }

    /**
     * @param  int|bool $store_id
     * @return string
     */
    public function getApiKey($store_id = false)
    {
        if ($this->isProdMode($store_id)) {
            return $this->encryptor->decrypt($this->getConfigValue(self::API_KEY, $store_id));
        }
        return $this->encryptor->decrypt($this->getConfigValue(self::API_KEY_SANDBOX, $store_id));
    }

    /**
     * @param  int|bool $store_id
     * @return string
     */
    public function getApiUrl($store_id = false)
    {
        if ($this->isProdMode($store_id)) {
            return ConstantValue::URL_PROD;
        }
        return ConstantValue::URL_SANDBOX;
    }

     /**
      * @return string
      */
    public function getMgmtApiUrl()
    {
        if ($this->isProdMode()) {
            return ConstantValue::MGMT_URL_PROD;
        }
        return ConstantValue::MGMT_URL_SANDBOX;
    }

    /**
     * @param  string   $path
     * @param  int|bool $store_id
     * @return mixed
     */
    public function getConfigValue($path, $store_id = false)
    {
        if ($store_id) {
            return $this->scopeConfig->getValue(
                $path,
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
                (string)$store_id
            );
        }
        return $this->scopeConfig->getValue(
            $path,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @param  string $checkoutCurrency
     * @return bool
     */
    public function checkCurrencySupport($checkoutCurrency)
    {
        return in_array($checkoutCurrency, ConstantValue::SUPPORTED_CURRENCIES);
    }
}
