<?php

namespace Revolut\Payment\Controller\Webhook;

use Magento\Framework\App\Action\Action;
use Revolut\Payment\Helper\DataHelper;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Sales\Model\Order;
use Revolut\Payment\Model\ConstantValue;
use Revolut\Payment\Helper\ConfigHelper;

/**
 * Class UpdateOrder
 *
 * @package Revolut\Payment\Controller\Webhook
 */
class RevolutWebhookController extends Action
{
    /**
     * @var \Magento\Framework\Json\Helper\Data
     */
    protected $jsonHelper;

    /**
     * @var ConfigHelper
     */
    protected $configHelper;

    /**
     * @var DataHelper
     */
    protected $dataHelper;

    /**
     * @var \Revolut\Payment\Model\RevolutOrderFactory
     */
    protected $revolutOrderFactory;

    /**
     * @var \Revolut\Payment\Model\ResourceModel\RevolutOrder
     */
    protected $revolutOrderResource;

    /**
     * @var \Revolut\Payment\Helper\Logger
     */
    protected $loggerRevolut;

    /**
     * @var \Magento\Sales\Api\Data\OrderInterface
     */
    protected $orderInterface;

    /**
     * @var \Magento\Sales\Model\Service\InvoiceService
     */
    protected $invoiceService;

    /**
     * @var \Magento\Framework\DB\TransactionFactory
     */
    protected $transactionFactory;

    /**
     * UpdateOrder constructor.
     *
     * @param \Magento\Framework\Json\Helper\Data               $jsonHelper
     * @param \Revolut\Payment\Helper\ConfigHelper              $configHelper         ,
     * @param \Revolut\Payment\Helper\DataHelper                $dataHelper           ,
     * @param \Revolut\Payment\Model\RevolutOrderFactory        $revolutOrderFactory
     * @param \Revolut\Payment\Model\ResourceModel\RevolutOrder $revolutOrderResource
     * @param \Revolut\Payment\Helper\Logger                    $loggerRevolut
     * @param \Magento\Sales\Api\Data\OrderInterface            $orderInterface
     * @param \Magento\Sales\Model\Service\InvoiceService       $invoiceService
     * @param \Magento\Framework\DB\TransactionFactory          $transactionFactory
     * @param Context                                           $context
     */
    public function __construct(
        \Magento\Framework\Json\Helper\Data               $jsonHelper,
        \Revolut\Payment\Helper\ConfigHelper              $configHelper,
        \Revolut\Payment\Helper\DataHelper                $dataHelper,
        \Revolut\Payment\Model\RevolutOrderFactory        $revolutOrderFactory,
        \Revolut\Payment\Model\ResourceModel\RevolutOrder $revolutOrderResource,
        \Revolut\Payment\Helper\Logger                    $loggerRevolut,
        \Magento\Sales\Api\Data\OrderInterface            $orderInterface,
        \Magento\Sales\Model\Service\InvoiceService       $invoiceService,
        \Magento\Framework\DB\TransactionFactory          $transactionFactory,
        Context                                           $context
    ) {
        $this->jsonHelper = $jsonHelper;
        $this->configHelper = $configHelper;
        $this->dataHelper = $dataHelper;
        $this->revolutOrderFactory = $revolutOrderFactory;
        $this->revolutOrderResource = $revolutOrderResource;
        $this->loggerRevolut = $loggerRevolut;
        $this->orderInterface = $orderInterface;
        $this->invoiceService = $invoiceService;
        $this->transactionFactory = $transactionFactory;
        parent::__construct($context);
    }

    /**
     * @return \Magento\Framework\App\ResponseInterface|\Magento\Framework\Controller\ResultInterface|void|null
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function execute()
    {
        $response = $this->_request->getContent();

        if (empty($response)) {
            return null;
        }

        $response = (array)$this->jsonHelper->jsonDecode($response);

        if (empty($response['order_id'])) {
            $this->loggerRevolut->debug("webhook event:");
            $this->loggerRevolut->debug($this->_request->getContent());
            return null;
        }

        $revolutOrderModel = $this->revolutOrderFactory->create();
        $this->revolutOrderResource->load($revolutOrderModel, $response['order_id'], 'revolut_order_id');
        $increment_order_id = $revolutOrderModel->getData('increment_order_id');

        if (empty($increment_order_id)) {
            return null;
        }
        /* @var Order $order */
        $order = $this->orderInterface->loadByIncrementId($increment_order_id);

        //if order is paid
        if (!$order->getTotalDue()) {
            return null;
        }

        $payment = $order->getPayment();

        if ($response['event'] == ConstantValue::WEBHOOK_EVENT_ORDER_COMPLETED) {
            $payment->setAdditionalInformation('webhook_event_order_completed', true);
            $payment->setIsTransactionPending(false);
            $payment->setTransactionId($response['order_id']);
            $payment->setShouldCloseParentTransaction(true);
            $payment->setIsTransactionClosed(true);
            $payment->setIsTransactionPending(false);

            $invoice = $this->invoiceService->prepareInvoice($order);
            $invoice->setRequestedCaptureCase(\Magento\Sales\Model\Order\Invoice::CAPTURE_ONLINE);
            $invoice->register();
            $invoice->getOrder()->setCustomerNoteNotify(0);
            $invoice->getOrder()->setIsInProcess(false);
            $invoice->setCanVoidFlag(0);
            $order->addStatusHistoryComment('An Invoice has been automatically generated', false);
            $transactionSave = $this->transactionFactory->create()->addObject($invoice)->addObject($invoice->getOrder());
            $transactionSave->save();
        }
    }
}
