<?php

namespace Revolut\Payment\Block\System\Config;

use Magento\Config\Block\System\Config\Form\Field;
use Magento\Backend\Block\Template\Context;
use Magento\Framework\Data\Form\Element\AbstractElement;
use Revolut\Payment\Model\ConstantValue;

/**
 * Class Webhook
 *
 * @package Revolut\Payment\Block\System\Config
 */
class Webhook extends Field
{
    /**
     * @var string
     */
    protected $_template = 'Revolut_Payment::system/config/webhook_button.phtml';

    /**
     * @var \Revolut\Payment\Helper\ConfigHelper
     */
    protected $configHelper;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \Revolut\Payment\Helper\DataHelper
     */
    protected $dataHelper;

    /**
     * @var \Magento\Framework\Json\Helper\Data
     */
    protected $jsonHelper;

    /**
     * @var \Revolut\Payment\Helper\Logger
     */
    public $loggerRevolut;

    /**
     * Webhook constructor.
     *
     * @param Context $context
     * @param array   $data
     */
    public function __construct(
        Context $context,
        \Revolut\Payment\Helper\ConfigHelper $configHelper,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Revolut\Payment\Helper\DataHelper $dataHelper,
        \Magento\Framework\Json\Helper\Data $jsonHelper,
        \Revolut\Payment\Helper\Logger $loggerRevolut,
        array $data = []
    ) {
        $this->configHelper = $configHelper;
        $this->storeManager = $storeManager;
        $this->dataHelper = $dataHelper;
        $this->jsonHelper = $jsonHelper;
        $this->loggerRevolut = $loggerRevolut;
        $this->checkApplePayDomainRegistration();
        parent::__construct($context, $data);
    }

    /**
     * @param  AbstractElement $element
     * @return string
     */
    public function render(AbstractElement $element)
    {
        $element->unsScope()->unsCanUseWebsiteValue()->unsCanUseDefaultValue();
        return parent::render($element);
    }

    /**
     * @param  AbstractElement $element
     * @return string
     */
    protected function _getElementHtml(AbstractElement $element)
    {
        return $this->_toHtml();
    }

    /**
     * @return mixed
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getButtonHtml()
    {
        $button = $this->getLayout()->createBlock(
            'Magento\Backend\Block\Widget\Button'
        )->setData(
            [
                'id' => 'webhook-config',
                'label' => __('Setup Webhook'),
            ]
        );
        
        return $button->toHtml();
    }

    /**
     * @return bool
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function checkSetupWebhook()
    {
        $mageWebHookUrl = $this->storeManager->getStore()->getBaseUrl() . ConstantValue::MAGE_URI_WEBHOOK;
        return $this->dataHelper->checkRevolutWebhookSetup($mageWebHookUrl);
    }

    /**
     * @return bool
     */
    public function checkApplePayDomainRegistration()
    {
        if (!$this->configHelper->checkApplePayDomainRegistration()) {
            return false;
        }
        
        try {
            $onboarding_file_path = $this->downloadOnboardingFile();
            $mageDomain = parse_url($this->storeManager->getStore()->getBaseUrl());
            $mageDomain = $mageDomain['host']; // @phpstan-ignore-line
            $register_result = $this->dataHelper->registerApplePayDomain($mageDomain);
            $register_res = \json_decode($register_result, true);
            \unlink($onboarding_file_path[0]);
            \unlink($onboarding_file_path[1]);
            if (!empty($register_res) && is_array($register_res) && in_array('code', array_keys($register_res))
            ) {
                $this->loggerRevolut->debug(json_encode($register_result));
                throw new \Exception('Cannot onboard Apple pay merchant: ' . $register_res['code']);
            }

            $this->loggerRevolut->debug("register result:");
            $this->loggerRevolut->debug($mageDomain);
            $this->loggerRevolut->debug(json_encode($register_result));
            $this->configHelper->setApplePayDomainConfig("done");
        } catch (\Exception $e) {
            $this->loggerRevolut->debug("Can not register apple pay domain {$e->getMessage()}");
            return false;
        }

        return true;
    }
    
    /**
     * @return array
     */
    public function downloadOnboardingFile()
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $directory = $objectManager->get('\Magento\Framework\Filesystem\DirectoryList');
        
        $rootPath = $directory->getRoot(); // @phpstan-ignore-line
        $pubPath  = $directory->getPath('pub'); // @phpstan-ignore-line
        $domain_onboarding_file_name = 'apple-developer-merchantid-domain-association';
        $domain_onboarding_file_directory = '.well-known';

        $onboarding_file_dir = $rootPath . '/' . $domain_onboarding_file_directory;
        $onboarding_file_path = $rootPath . '/' . $domain_onboarding_file_directory . '/' . $domain_onboarding_file_name;
        
        $onboarding_file_dir_pub = $pubPath . '/' . $domain_onboarding_file_directory;
        $onboarding_file_path_pub = $onboarding_file_dir_pub . '/' . $domain_onboarding_file_name;
        
        $domain_onboarding_file_remote_link = 'https://assets.revolut.com/api-docs/merchant-api/files/apple-developer-merchantid-domain-association';
        
        if (!\file_exists($onboarding_file_dir) && ! mkdir($onboarding_file_dir, 0755)) {
            throw new \Exception("Can not locate onboarding file: permission issue");
        }

        if (!\file_exists($onboarding_file_dir_pub) && ! mkdir($onboarding_file_dir_pub, 0755)) {
            throw new \Exception("Can not locate onboarding file: pub dir permission issue");
        }

        if (!\file_put_contents(
            $onboarding_file_path,
            \file_get_contents($domain_onboarding_file_remote_link)
        )
        ) {
            throw new \Exception('Cannot onboard Apple pay merchant: Can not locate on-boarding file');
        }

        if (!\file_put_contents(
            $onboarding_file_path_pub,
            \file_get_contents($domain_onboarding_file_remote_link)
        )
        ) {
            throw new \Exception('Cannot onboard Apple pay merchant: Can not locate on-boarding file in pub dir');
        }

        return [$onboarding_file_path, $onboarding_file_path_pub];
    }
}
